package cn.nexgo.inbas.components.others.print;

import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.os.Environment;
import android.text.TextUtils;

import com.nexgo.common.FileUtils;
import com.nexgo.oaf.apiv3.APIProxy;
import com.nexgo.oaf.apiv3.SdkResult;
import com.nexgo.oaf.apiv3.device.printer.AlignEnum;
import com.nexgo.oaf.apiv3.device.printer.DotMatrixFontEnum;
import com.nexgo.oaf.apiv3.device.printer.FontEntity;
import com.nexgo.oaf.apiv3.device.printer.OnPrintListener;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;

import cn.nexgo.inbas.R;
import cn.nexgo.inbas.common.Constans.CardType;
import cn.nexgo.inbas.common.GData;
import cn.nexgo.inbas.components.data.bean.TransRecordEntity;
import cn.nexgo.inbas.controller.global.TransType;
import cn.nexgo.utils.BaseUtils;
import cn.nexgo.utils.FormatUtils;

/**
 * Printer
 * Created by lee on 2019/6/10.
 */

public class Printer {
    private static final FontEntity FONT_SMALL = new FontEntity(DotMatrixFontEnum.CH_SONG_20X20, DotMatrixFontEnum.ASC_SONG_8X16);
    private static final FontEntity FONT_NORMAL = new FontEntity(DotMatrixFontEnum.CH_SONG_24X24, DotMatrixFontEnum.ASC_SONG_12X24);
    private static final FontEntity FONT_BOLD = new FontEntity(DotMatrixFontEnum.CH_SONG_32X32, DotMatrixFontEnum.ASC_SONG_16X32);
    private static final FontEntity FONT_BIG = new FontEntity(DotMatrixFontEnum.CH_SONG_24X24, DotMatrixFontEnum.ASC_SONG_12X24, false, true);
    private static final int SPLIT_LEN = 3;
    private static final String SPLIT = ",";

    private static final com.nexgo.oaf.apiv3.device.printer.Printer mPrinter = APIProxy.getDeviceEngine(BaseUtils.getApp().getApplicationContext()).getPrinter();
    private static Logger mLog = LoggerFactory.getLogger(Printer.class.getSimpleName());
    private static PrintListener mListener;

    public static void startPrint(TransRecordEntity entity, boolean pushPaper, boolean isRePrint, CopyType copyType, PrintListener listener) {
        int result = mPrinter.initPrinter();
        if (result != SdkResult.Success) {
            listener.onFail(ErrorType.OTHER);
            return;
        }
        if (mPrinter.getStatus() == SdkResult.Printer_PaperLack) {
            listener.onFail(ErrorType.OUT_OF_PAPER);
            return;
        }
        if (mPrinter.getStatus() == SdkResult.Printer_TooHot) {
            listener.onFail(ErrorType.OVER_HEAT);
            return;
        }
        if (mPrinter.getStatus() != SdkResult.Success) {
            listener.onFail(ErrorType.OTHER);
            return;
        }
        mListener = listener;
        formHeader();
        formPrintTransData(entity);
        formFooter(entity, isRePrint, copyType);
        mPrinter.startPrint(pushPaper, onPrintListener);
    }

    private static OnPrintListener onPrintListener = new OnPrintListener() {
        @Override
        public void onPrintResult(int retCode) {
            if (retCode == SdkResult.Success) {  // print success
                mListener.onSuccess();
            } else {
                ErrorType errorType;
                switch (retCode) {
                    case SdkResult.Printer_PaperLack:
                        errorType = ErrorType.OUT_OF_PAPER;
                        break;
                    case SdkResult.Printer_TooHot:
                        errorType = ErrorType.OVER_HEAT;
                        break;
                    default:
                        errorType = ErrorType.OTHER;
                        break;
                }
                mListener.onFail(errorType);
            }
        }
    };

    private static void formPrintTransData(TransRecordEntity transRecord) {
        //Merchant name
        mPrinter.appendPrnStr(getString(R.string.print_merchant_name), FONT_NORMAL, AlignEnum.LEFT);
        mPrinter.appendPrnStr(getNonNull(GData.getInstance().getSetupEntity().getTermMerchName()), FONT_NORMAL, AlignEnum.RIGHT);
        //MID
        mPrinter.appendPrnStr(getString(R.string.print_mid), getNonNull(transRecord.getMsgMerchID()), FONT_NORMAL);
        //TID
        mPrinter.appendPrnStr(getString(R.string.print_tid), getNonNull(transRecord.getMsgTermID()), FONT_NORMAL);
        //Cashier ID
        mPrinter.appendPrnStr(getString(R.string.print_cashier_no), getNonNull(transRecord.getMsgCashierNO()), FONT_NORMAL);
        //Issuer bank NO.
        mPrinter.appendPrnStr(getString(R.string.print_issuer_no), getNonNull(transRecord.getMsgIssuerCode()), FONT_NORMAL);
        //Acquirer bank NO.
        mPrinter.appendPrnStr(getString(R.string.print_acquier_no), getNonNull(transRecord.getMsgAcquirerCode()), FONT_NORMAL);
        //-----card NO.
        mPrinter.appendPrnStr(getString(R.string.print_card_no), FONT_NORMAL, AlignEnum.LEFT);
        String strTmp = getNonNull(transRecord.getCardNoEnc());
        if (GData.getInstance().getSetupEntity().getTransMaskPAN()) {
            strTmp = FormatUtils.maskString(strTmp, 6, 4, '*');
        }
        strTmp += " ";
        switch (transRecord.getCardTypeEnum()) {
            case MANUAL:
                strTmp += getString(R.string.print_card_manual);
                break;
            case MAG:
                strTmp += getString(R.string.print_card_mag);
                break;
            case ICC:
                strTmp += getString(R.string.print_card_icc);
                break;
            case CTLS:
                strTmp += getString(R.string.print_card_ctls);
                break;
            default:
                strTmp += getString(R.string.print_card_none);
                break;
        }
        mLog.debug("card number:{}", strTmp);
        mPrinter.appendPrnStr(strTmp, FONT_NORMAL, AlignEnum.RIGHT);

        //-----expired date
        if (transRecord.getCardExpDateEnc() == null || transRecord.getCardExpDateEnc().length() < 4) {
            strTmp = " ";
        } else {
            // YY/MM
            strTmp = transRecord.getCardExpDateEnc().substring(0, 2) + "/";
            strTmp += transRecord.getCardExpDateEnc().substring(2, 4);
        }
        mPrinter.appendPrnStr(getString(R.string.print_expired_date), strTmp, FONT_NORMAL);

        //-----Transaction Type
        strTmp = getTransName(transRecord.getTransTypeEnum());
        mPrinter.appendPrnStr(getString(R.string.print_trans_type), FONT_NORMAL, AlignEnum.LEFT);
        FONT_BOLD.setBold(true);
        mPrinter.appendPrnStr(strTmp, FONT_BOLD, AlignEnum.RIGHT);

        //-----batch NO.
        mPrinter.appendPrnStr(getString(R.string.print_batch_no), getNonNull(transRecord.getMsgBatchNO()), FONT_NORMAL);
        //-----Trace NO.
        mPrinter.appendPrnStr(getString(R.string.print_trace_no), getNonNull(transRecord.getMsgTraceNO()), FONT_NORMAL);
        //-----authorization NO.
        mPrinter.appendPrnStr(getString(R.string.print_auth_no), getNonNull(transRecord.getMsgAuthCode()), FONT_NORMAL);
        //-----reference NO.
        mPrinter.appendPrnStr(getString(R.string.print_reference_no), getNonNull(transRecord.getMsgRefNO()), FONT_NORMAL);
        //-----date & time
        formDateTime(transRecord);

        //-----amount
        mPrinter.appendPrnStr(getString(R.string.print_amount), FONT_NORMAL, AlignEnum.LEFT);
        strTmp = getNonNull(GData.getInstance().getSetupEntity().getTermCurrencyName()) + " ";

        if (GData.getInstance().getSetupEntity().getTransPrintMinus()
                && (transRecord.getTransTypeEnum() == TransType.VOID
                || transRecord.getTransTypeEnum() == TransType.REFUND)) {
            strTmp += "-";
        }
        strTmp += FormatUtils.formatAmount(getNonNull(transRecord.getMsgAmount()),
                SPLIT_LEN, SPLIT, GData.getInstance().getSetupEntity().getTermAmountDecimalLen());

        mPrinter.appendPrnStr(strTmp, FONT_BOLD, AlignEnum.CENTER);

        //Balacnce
        if (!TextUtils.isEmpty(transRecord.getMsgBalance()) && transRecord.getMsgBalance().length() > 1) {
            strTmp = getNonNull(GData.getInstance().getSetupEntity().getTermCurrencyName()) + " ";
            if (transRecord.getMsgBalance().substring(0, 1).equals("D")) {
                strTmp += "-";
            }
            strTmp += FormatUtils.formatAmount(getNonNull(transRecord.getMsgBalance()).substring(1),
                    SPLIT_LEN, SPLIT, GData.getInstance().getSetupEntity().getTermAmountDecimalLen());
            mPrinter.appendPrnStr(getString(R.string.print_balance), strTmp, FONT_NORMAL);
        }

        //-----International credit card organization
        if (!TextUtils.isEmpty(transRecord.getMsgInterOrgCode())) {
            mPrinter.appendPrnStr(getString(R.string.print_card_organization), transRecord.getMsgInterOrgCode(), FONT_NORMAL);

        }
    }

    private static void formHeader() {
        Bitmap bitmap = getBitmapFromFile("nexgo_logo.png");
        if (bitmap == null) {
            mPrinter.appendPrnStr(getString(R.string.print_nexgo), FONT_BIG, AlignEnum.CENTER);
        } else {
            mPrinter.appendImage(bitmap, AlignEnum.CENTER);
        }
    }

    /**
     * Form general footer bitmap for receipt
     */
    private static void formFooter(TransRecordEntity entity, boolean isRePrint, CopyType copyType) {
        mPrinter.appendPrnStr("-----------------------------", FONT_NORMAL, AlignEnum.LEFT);
        formRefInfo(entity);
        if (isRePrint) {
            mPrinter.appendPrnStr(getString(R.string.print_reprint_flag), FONT_NORMAL, AlignEnum.CENTER);
        }

        if (!TextUtils.isEmpty(entity.getCardHolderNameEnc())) {
            mPrinter.appendPrnStr(getString(R.string.print_cardholder) + getNonNull(entity.getCardHolderNameEnc()), FONT_SMALL, AlignEnum.LEFT);
        }
        if (copyType == CopyType.MERCHANT) {
            formSignature();
        }
        mPrinter.appendPrnStr("-----------------------------", FONT_NORMAL, AlignEnum.LEFT);
        mPrinter.appendPrnStr(getString(R.string.print_terminal_info), FONT_SMALL, AlignEnum.LEFT);
    }

    /**
     * Form signature content bitmap for receipt
     */
    private static void formSignature() {
        mPrinter.appendPrnStr(getString(R.string.print_signature), FONT_NORMAL, AlignEnum.LEFT);
        mPrinter.appendPrnStr(" ", FONT_NORMAL, AlignEnum.LEFT);
        // FIXME: 2018/8/7 Hassan add line
        mPrinter.appendPrnStr("______________________", FONT_NORMAL, AlignEnum.CENTER);
        mPrinter.appendPrnStr(getString(R.string.print_footer_info), FONT_SMALL, AlignEnum.LEFT);
    }

    /**
     * Form reference information bitmap for receipt
     */
    private static void formRefInfo(TransRecordEntity entity) {
        String strtmp;
        TransType transType = entity.getTransTypeEnum();

        //-----reference
        mPrinter.appendPrnStr(getString(R.string.print_reference), FONT_NORMAL, AlignEnum.LEFT);

        //-----orig trace No.
        if (transType == TransType.VOID || transType == TransType.TIP) {
            mPrinter.appendPrnStr(getString(R.string.print_org_trace_no), getNonNull(entity.getMsgOrigTraceNO()), FONT_NORMAL);
            mPrinter.appendPrnStr(getString(R.string.print_org_batch_no), getNonNull(entity.getMsgOrigBatchNO()), FONT_NORMAL);

        }

        //-----orig ref no. & orig date
        if (transType == TransType.REFUND) {
            if (!TextUtils.isEmpty(entity.getMsgRefNO())) {
                mPrinter.appendPrnStr(getString(R.string.print_org_ref_no), entity.getMsgRefNO(), FONT_NORMAL);
            }
        }
        if (transType == TransType.REFUND || transType == TransType.TIP) {
            if (!TextUtils.isEmpty(entity.getMsgOrigDate())) {
                strtmp = entity.getMsgOrigDate().substring(0, 2) + "/" + entity.getMsgOrigDate().substring(2, 4);
                mPrinter.appendPrnStr(getString(R.string.print_org_date), strtmp, FONT_NORMAL);
            }
        }
        if (!TextUtils.isEmpty(entity.getMsgIssuerInfo())) {
            mPrinter.appendPrnStr(entity.getMsgIssuerInfo(), FONT_SMALL, AlignEnum.LEFT);
        }
        if (!TextUtils.isEmpty(entity.getMsgCUPInfo())) {
            mPrinter.appendPrnStr(entity.getMsgCUPInfo(), FONT_SMALL, AlignEnum.LEFT);
        }
        if (!TextUtils.isEmpty(entity.getMsgAcqInfo())) {
            mPrinter.appendPrnStr(entity.getMsgAcqInfo(), FONT_SMALL, AlignEnum.LEFT);
        }
        if (!TextUtils.isEmpty(entity.getMsgTermReserved())) {
            mPrinter.appendPrnStr(entity.getMsgTermReserved(), FONT_SMALL, AlignEnum.LEFT);
        }

//        //ICC reference
        if (transType == TransType.SALE &&
                (entity.getCardTypeEnum() == CardType.ICC || entity.getCardTypeEnum() == CardType.CTLS)) {
            if (!TextUtils.isEmpty(entity.getEmvAC())) {
                if (entity.getStatusOffline()) {
                    mPrinter.appendPrnStr(getString(R.string.print_icc_tc) + entity.getEmvAC(), FONT_SMALL, AlignEnum.LEFT);
                } else {
                    mLog.debug("ARQC:{}", entity.getEmvAC());
                    mPrinter.appendPrnStr(getString(R.string.print_icc_arqc) + entity.getEmvAC(), FONT_SMALL, AlignEnum.LEFT);
                }
            }

            if (!TextUtils.isEmpty(entity.getEmvTVR())) {
                strtmp = getString(R.string.print_icc_tvr) + entity.getEmvTVR();
                if (!TextUtils.isEmpty(String.valueOf(entity.getEmvPanSN()))) {
                    strtmp += "  " + getString(R.string.print_icc_csn) + String.valueOf(entity.getEmvPanSN());
                }
                mLog.debug("TVR:{}", strtmp);
                mPrinter.appendPrnStr(strtmp, FONT_SMALL, AlignEnum.LEFT);
            }
            if (!TextUtils.isEmpty(entity.getEmvAID())) {
                mPrinter.appendPrnStr(getString(R.string.print_icc_aid) + entity.getEmvAID(), FONT_SMALL, AlignEnum.LEFT);
            }

        }

        mPrinter.appendPrnStr(" ", FONT_SMALL, AlignEnum.LEFT);
    }

    /**
     * Form date/time bitmap for receipt format: xxxx/xx/xx xx:xx:xx
     */
    private static void formDateTime(TransRecordEntity entity) {
        String now = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss", Locale.ENGLISH).format(new Date());
        String strTmp;
        if (entity.getMsgTransDate() != null && entity.getMsgTransDate().length() >= 4
                && entity.getMsgTransTime() != null && entity.getMsgTransTime().length() >= 6) {
            strTmp = now.substring(0, 4) + "/";     //YYYY
            strTmp += entity.getMsgTransDate().substring(0, 2) + "/";    //MM
            strTmp += entity.getMsgTransDate().substring(2, 4) + " ";    //DD
            strTmp += entity.getMsgTransTime().substring(0, 2) + ":";    //hh
            strTmp += entity.getMsgTransTime().substring(2, 4) + ":";    //mm
            strTmp += entity.getMsgTransTime().substring(4, 6);          //ss
        } else {
            strTmp = "";
        }
        mPrinter.appendPrnStr(getString(R.string.print_date_time), strTmp, FONT_NORMAL);
    }

    // FIXME: 2018/8/7 Hassan modify print logo
    private static Bitmap getBitmapFromFile(String filename) {
        Bitmap bitmap = null;
        InputStream is;
        String logoFile = Environment.getExternalStorageDirectory().getPath() + "/" + BaseUtils.getApp().getPackageName() +
                "/" + filename;

        try {
            if (FileUtils.isFileExists(logoFile)) {
                is = new FileInputStream(logoFile);
            } else {
                is = BaseUtils.getApp().getAssets().open(filename);
            }
            if (is == null) {
                return null;
            }
            bitmap = BitmapFactory.decodeStream(is);
        } catch (IOException e) {
            e.printStackTrace();
        }

        return bitmap;
    }

    /**
     * print listener
     */
    public interface PrintListener {
        void onSuccess();

        void onFail(Printer.ErrorType error);
    }

    /**
     * error type
     */
    public enum ErrorType {
        OVER_HEAT,
        OUT_OF_PAPER,
        OTHER
    }

    public static String errorType2Str(Printer.ErrorType errorType) {
        switch (errorType) {
            case OUT_OF_PAPER:
                return BaseUtils.getApp().getString(R.string.printererror_outofpaper);
            case OVER_HEAT:
                return BaseUtils.getApp().getString(R.string.printererror_overheat);
        }
        return BaseUtils.getApp().getString(R.string.printererror_othererror);
    }

    private static String getString(int stringId) {
        return BaseUtils.getApp().getString(stringId);
    }

    private static String getTransName(TransType transType) {
        return BaseUtils.getApp().getApplicationContext().getString(transType.getName());
    }

    private static String getNonNull(String string) {
        if (string == null || string.length() == 0) {
            return " ";
        }
        return string;
    }

    /**
     * copy type
     */
    public enum CopyType {
        NONE,
        MERCHANT,
        CARDHODLDER,
        BANK,
    }
}
