package cn.nexgo.inbas.components.data.repository;

import android.database.Cursor;
import android.support.annotation.NonNull;

import org.greenrobot.greendao.DaoException;
import org.greenrobot.greendao.query.QueryBuilder;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Locale;

import cn.nexgo.inbas.common.Constans.CardType;
import cn.nexgo.inbas.components.data.DataConstant;
import cn.nexgo.inbas.components.data.DataOpenHelper;
import cn.nexgo.inbas.components.data.bean.TransRecordEntity;
import cn.nexgo.inbas.components.data.daoGenerate.DaoSession;
import cn.nexgo.inbas.components.data.daoGenerate.TransRecordEntityDao;
import cn.nexgo.inbas.components.data.helper.TransRecordDbHelper;
import cn.nexgo.inbas.controller.global.TransType;
import cn.nexgo.utils.AppLogUtils;

/***************************************************************************************************
 *                                  Copyright (C), Nexgo Inc.                                      *
 *                                    http://www.nexgo.cn                                          *
 ***************************************************************************************************
 * File Name     : TransRecordRepository.java
 * Usage         : 
 * Version       : 1
 * Author        : Brad
 * Date          : 2017/12/13 
 * Modification  : Created file

 **************************************************************************************************/
public class TransRecordRepository {
    private static String LOG_TAG = "TransRecordRepository";

    private TransRecordEntityDao recordEntityDao;

    private final String SQL_TRUE = "1";
    private final String SQL_FALSE = "0";


    private static volatile TransRecordRepository INSTANCE;

    private TransRecordRepository() {
        init();
    }

    public static TransRecordRepository getInstance() {
        if (INSTANCE == null) {
            synchronized (TransRecordRepository.class) {
                if (INSTANCE == null) {
                    INSTANCE = new TransRecordRepository();
                }
            }
        }
        return INSTANCE;
    }

    /**
     * Initialize Entity Dao
     */
    private void init() {
        DaoSession daoSession = TransRecordDbHelper.getInstance().getDaoSession();
        recordEntityDao = daoSession.getTransRecordEntityDao();
    }

    /**
     * Insert an Entity. If id is already existed, then update the existed record.
     *
     * @param entity TransRecordEntity
     * @return Success & fail
     */
    public boolean insert(TransRecordEntity entity) {
        if (recordEntityDao == null) {
            init();
        }
        if (entity != null) {
            recordEntityDao.insert(entity);
//            DataLogUtils.trace("Inserted id = {}", entity.getId());
            return true;
        }
        return false;
    }

    /**
     * Delete an Entity
     *
     * @param entity TransRecordEntity
     * @return Success & fail
     */
    public boolean delete(TransRecordEntity entity) {
        if (recordEntityDao == null) {
            init();
        }
        if (entity != null) {
            try {
                recordEntityDao.delete(entity);
            } catch (DaoException e) {
                AppLogUtils.error(true, LOG_TAG, "Delete failed throw Dao exception");
                return false;
            }
            return true;
        }
        return false;
    }

    /**
     * Delete Last Entity
     *
     * @return Success & fail
     */
    public boolean deleteLast() {
        if (recordEntityDao == null) {
            init();
        }
        recordEntityDao.detachAll();
        QueryBuilder<TransRecordEntity> qb = recordEntityDao.queryBuilder().orderDesc(TransRecordEntityDao.Properties.Id).limit(1);    //invert list
        List<TransRecordEntity> gotRecords = qb.build().list();      //Get Object List
        if (gotRecords == null || gotRecords.isEmpty()) {
            return true;
        }
        delete(gotRecords.get(0));
        return true;
    }

    /**
     * Delete All Entities
     */
    public void deleteAll() {
        if (recordEntityDao == null) {
            init();
        }
        recordEntityDao.deleteAll();
    }

    /**
     * Get Last Entity
     *
     * @return Entity
     */
    public TransRecordEntity getLast() {
        if (recordEntityDao == null) {
            init();
        }
        recordEntityDao.detachAll();
        QueryBuilder<TransRecordEntity> qb = recordEntityDao.queryBuilder().orderDesc(TransRecordEntityDao.Properties.Id).limit(1);    //invert list
        return qb.build().unique();
    }

    /**
     * Update an Entity. If id is not existed, insert a new entity.
     *
     * @param entity TransRecordEntity
     * @return Success & fail
     */
    public boolean update(TransRecordEntity entity) {
        if (recordEntityDao == null) {
            init();
        }
        if (entity != null) {
            recordEntityDao.update(entity);
            return true;
        }
        return false;
    }

    public boolean update(List<TransRecordEntity> entitys) {
        if (recordEntityDao == null) {
            init();
        }
        if (entitys != null) {
            recordEntityDao.updateInTx(entitys);
            return true;
        }
        return false;
    }

    /**
     * Get Record Numbers
     *
     * @return Record Numbers
     */
    public int getRecordNum() {
        if (recordEntityDao == null) {
            init();
        }
        //DataLogUtils.trace("getRecordNum = "+ (int) recordEntityDao.count());
        return (int) recordEntityDao.count();
    }

    public int getOfflineRecordNum() {
        if (recordEntityDao == null) {
            init();
        }
        recordEntityDao.detachAll();

        int tipCount = (int) recordEntityDao.queryBuilder().where(TransRecordEntityDao.Properties.TransID.eq(TransType.TIP.ordinal()),
                TransRecordEntityDao.Properties.StatusOffineUpload.eq(DataConstant.OffUploadStatus.OFFLINE_NONE.ordinal()))
                .count();

        int iccOfflineCount = (int) recordEntityDao.queryBuilder().where(TransRecordEntityDao.Properties.StatusOffline.eq(true),
                TransRecordEntityDao.Properties.StatusOffineUpload.eq(DataConstant.OffUploadStatus.OFFLINE_NONE.ordinal()))
                .count();

        return (tipCount + iccOfflineCount);
    }

    /**
     * Get an entity by trace number
     *
     * @param traceNo Trace number
     * @return if More than one record are found, just return the first one.
     */
    public TransRecordEntity getByTraceNo(String traceNo) {
        if (recordEntityDao == null) {
            init();
        }
        recordEntityDao.detachAll();
        QueryBuilder<TransRecordEntity> qb = recordEntityDao.queryBuilder();
        qb.where(TransRecordEntityDao.Properties.MsgTraceNO.eq(traceNo))
                .limit(10);     // Limit 
        List<TransRecordEntity> gotRecords = qb.build().list();
        if (gotRecords == null || gotRecords.isEmpty()) {
            AppLogUtils.warn(true, LOG_TAG, "No Records Found By TraceNo");
            return null;
        }
        if (gotRecords.size() > 1) {
            AppLogUtils.warn(true, LOG_TAG, "More Than One Record Found");
            //return null;
        }
        return gotRecords.get(0);
    }

    public TransRecordEntity getSuccessAndOnlineSaleTrans(String traceNo) {
        if (recordEntityDao == null) {
            init();
        }
        recordEntityDao.detachAll();
        QueryBuilder<TransRecordEntity> qb = recordEntityDao.queryBuilder();
        return qb.where(TransRecordEntityDao.Properties.MsgTraceNO.eq(traceNo),
                TransRecordEntityDao.Properties.StatusOffline.eq(false),
                TransRecordEntityDao.Properties.StatusIccFailTrans.eq(false),
                TransRecordEntityDao.Properties.StatusNeedReversal.eq(false))
                .unique();     // Limit
    }

    /**
     * Get an Entity by record offset
     *
     * @param offset the offset of existed records, not id.  range from 0~recordEntityDao.count().
     * @return entity
     */
    public TransRecordEntity getRecordByIndex(int offset) {
        if (recordEntityDao == null) {
            init();
        }
        recordEntityDao.detachAll();
        if (recordEntityDao.count() < offset + 1) {
            AppLogUtils.warn(true, LOG_TAG, "No such more records, total = {}", recordEntityDao.count());
            return null;
        }
//        LazyList<TransRecordEntity> gotRecords = recordEntityDao.queryBuilder().build().listLazyUncached();
//        TransRecordEntity entity = gotRecords.get(offset);
//        gotRecords.close();  //Must call close()
        return recordEntityDao.queryBuilder().offset(offset).limit(1).unique();
    }

    /**
     * Get an Entity by id
     *
     * @param id id
     * @return entity
     */
    public TransRecordEntity getRecordById(Long id) {
        if (recordEntityDao == null) {
            init();
        }
        recordEntityDao.detachAll();
        return recordEntityDao.load(id);
    }


    /**
     * Get need reversal
     *
     * @return entity
     */
    public TransRecordEntity getNeedReversal(int offset) {
        if (recordEntityDao == null) {
            init();
        }
        recordEntityDao.detachAll();

        return recordEntityDao.queryBuilder()
                .where(TransRecordEntityDao.Properties.StatusNeedReversal.eq(true))
                .offset(offset).limit(1).unique();
    }

    /**
     * delete reversal
     *
     * @return true: success, false: failed
     */
    public boolean deleteReversal() {
        recordEntityDao.queryBuilder()
                .where(TransRecordEntityDao.Properties.StatusNeedReversal.eq(true)).buildDelete().executeDeleteWithoutDetachingEntities();
        return true;
    }

    /**
     * Get script uploading record
     *
     * @return entity
     */
    public TransRecordEntity getScriptUpload(int offset) {
        if (recordEntityDao == null) {
            init();
        }
        recordEntityDao.detachAll();
        QueryBuilder<TransRecordEntity> qb = recordEntityDao.queryBuilder();
        return qb.where(TransRecordEntityDao.Properties.StatusNeedScript.eq(true))
                .offset(offset).limit(1).unique();     // Limit
    }

    /**
     * Get offline transaction uploading record (Tip)
     *
     * @return entity
     */
    public TransRecordEntity getOfflineUploadTip(int offset) {
        if (recordEntityDao == null) {
            init();
        }
        recordEntityDao.detachAll();

        QueryBuilder<TransRecordEntity> qb = recordEntityDao.queryBuilder();

        return qb.where(TransRecordEntityDao.Properties.StatusOffineUpload.eq(DataConstant.OffUploadStatus.OFFLINE_NONE.ordinal()),
                TransRecordEntityDao.Properties.StatusOffline.eq(true),
                qb.or(TransRecordEntityDao.Properties.TransID.eq(TransType.ADJUST.ordinal()),
                        TransRecordEntityDao.Properties.TransID.eq(TransType.TIP.ordinal())))
                .offset(offset).limit(1).unique();
    }

    /**
     * Get ICC offline transaction uploading record (IC card)
     *
     * @return entity
     */
    public TransRecordEntity getOfflineUploadIcc(int offset) {
        if (recordEntityDao == null) {
            init();
        }
        recordEntityDao.detachAll();

        QueryBuilder<TransRecordEntity> qb = recordEntityDao.queryBuilder();
        return qb.where(TransRecordEntityDao.Properties.TransID.eq(TransType.SALE.ordinal()),
                TransRecordEntityDao.Properties.StatusOffineUpload.eq(DataConstant.OffUploadStatus.OFFLINE_NONE.ordinal()),
                TransRecordEntityDao.Properties.StatusOffline.eq(true),
                TransRecordEntityDao.Properties.StatusIccFailTrans.eq(false))
                .limit(1).offset(offset).unique();
    }

    /**
     * Search records by trace Number(like)
     *
     * @param traceNo trace number
     * @param limit   Max size
     * @param offset  offset
     * @return entities
     */
    public List<TransRecordEntity> searchTraceNoLike(String traceNo, int limit, int offset) {
        if (recordEntityDao == null) {
            init();
        }
        recordEntityDao.detachAll();
        if (traceNo == null) {
            return recordEntityDao.queryBuilder()
                    .where(TransRecordEntityDao.Properties.StatusIccFailTrans.eq(false),
                            TransRecordEntityDao.Properties.StatusNeedReversal.eq(false))
                    .limit(limit).offset(offset)
                    .orderDesc(TransRecordEntityDao.Properties.Id).list();
        }
        return recordEntityDao.queryBuilder().
                where(TransRecordEntityDao.Properties.MsgTraceNO.like("%" + traceNo + "%"),
                        TransRecordEntityDao.Properties.StatusIccFailTrans.eq(false),
                        TransRecordEntityDao.Properties.StatusNeedReversal.eq(false))
                .limit(limit).offset(offset)
                .orderDesc(TransRecordEntityDao.Properties.Id).list();

    }

    /**
     * Search records by trace Number(like) and date is today
     *
     * @param traceNo trace number
     * @param limit   Max size
     * @param offset  offset
     * @return entities
     */
    public List<TransRecordEntity> searchTraceNoLikeToday(String traceNo, int limit, int offset) {
        if (recordEntityDao == null) {
            init();
        }
        String date = getCurrentDate();

        recordEntityDao.detachAll();
        if (traceNo == null || traceNo.equals("")) {
            return recordEntityDao.queryBuilder()
                    .where(TransRecordEntityDao.Properties.MsgTransDate.eq(date),
                            TransRecordEntityDao.Properties.StatusIccFailTrans.eq(false),
                            TransRecordEntityDao.Properties.StatusNeedReversal.eq(false))
                    .limit(limit).offset(offset)
                    .orderDesc(TransRecordEntityDao.Properties.Id).list();
        }
        return recordEntityDao.queryBuilder()
                .where(TransRecordEntityDao.Properties.MsgTraceNO.like("%" + traceNo + "%"),
                        TransRecordEntityDao.Properties.MsgTransDate.eq(date),
                        TransRecordEntityDao.Properties.StatusIccFailTrans.eq(false),
                        TransRecordEntityDao.Properties.StatusNeedReversal.eq(false))
                .limit(limit).offset(offset)
                .orderDesc(TransRecordEntityDao.Properties.Id).list();

    }

    /**
     * Get the count of records by trace Number(like) and date is today
     *
     * @param traceNo trace number
     * @return count
     */
    public long getSearchTraceNoLikeCountToday(String traceNo) {
        if (recordEntityDao == null) {
            init();
        }
        String date = getCurrentDate();
        if (traceNo == null || traceNo.equals("")) {
            return recordEntityDao.queryBuilder()
                    .where(TransRecordEntityDao.Properties.MsgTransDate.eq(date),
                            TransRecordEntityDao.Properties.StatusIccFailTrans.eq(false),
                            TransRecordEntityDao.Properties.StatusNeedReversal.eq(false))
                    .count();
        }
        return recordEntityDao.queryBuilder()
                .where(TransRecordEntityDao.Properties.MsgTraceNO.like("%" + traceNo + "%"),
                        TransRecordEntityDao.Properties.MsgTransDate.eq(date),
                        TransRecordEntityDao.Properties.StatusIccFailTrans.eq(false),
                        TransRecordEntityDao.Properties.StatusNeedReversal.eq(false))
                .count();

    }

    /**
     * Search records by trace Number(like) and date is before today
     *
     * @param traceNo trace number
     * @param limit   Max size
     * @param offset  offset
     * @return entities
     */
    public List<TransRecordEntity> searchTraceNoLikeEarlier(String traceNo, int limit, int offset) {
        if (recordEntityDao == null) {
            init();
        }
        String date = getCurrentDate();

        recordEntityDao.detachAll();
        recordEntityDao.detachAll();
        if (traceNo == null || traceNo.equals("")) {
            return recordEntityDao.queryBuilder()
                    .where(TransRecordEntityDao.Properties.MsgTransDate.lt(date),
                            TransRecordEntityDao.Properties.StatusIccFailTrans.eq(false),
                            TransRecordEntityDao.Properties.StatusNeedReversal.eq(false))
                    .limit(limit).offset(offset)
                    .orderDesc(TransRecordEntityDao.Properties.Id).list();
        }
        return recordEntityDao.queryBuilder()
                .where(TransRecordEntityDao.Properties.MsgTraceNO.like("%" + traceNo + "%"),
                        TransRecordEntityDao.Properties.MsgTransDate.lt(date),
                        TransRecordEntityDao.Properties.StatusIccFailTrans.eq(false),
                        TransRecordEntityDao.Properties.StatusNeedReversal.eq(false))
                .limit(limit).offset(offset)
                .orderDesc(TransRecordEntityDao.Properties.Id).list();

    }

    /**
     * Get the count of records by trace Number(like) and date is before today
     *
     * @param traceNo trace number
     * @return count
     */
    public long getSearchTraceNoLikeCountEarlier(String traceNo) {
        if (recordEntityDao == null) {
            init();
        }
        String date = getCurrentDate();

        if (traceNo == null || traceNo.equals("")) {
            return recordEntityDao.queryBuilder()
                    .where(TransRecordEntityDao.Properties.MsgTransDate.lt(date),
                            TransRecordEntityDao.Properties.StatusIccFailTrans.eq(false),
                            TransRecordEntityDao.Properties.StatusNeedReversal.eq(false))
                    .count();
        }
        return recordEntityDao.queryBuilder()
                .where(TransRecordEntityDao.Properties.MsgTraceNO.like("%" + traceNo + "%"),
                        TransRecordEntityDao.Properties.MsgTransDate.lt(date),
                        TransRecordEntityDao.Properties.StatusIccFailTrans.eq(false),
                        TransRecordEntityDao.Properties.StatusNeedReversal.eq(false))
                .count();

    }

    /**
     * Search records by trace Number(like) and date is today, and specified transaction id
     *
     * @param traceNo trace number
     * @param transID transaction id
     * @param limit   Max size
     * @param offset  offset
     * @return entities
     */
    public List<TransRecordEntity> searchTraceNoLikeAndTranIdToday(String traceNo, TransType transID, int limit, int offset) {
        recordEntityDao.detachAll();
        String date = getCurrentDate();

        if (traceNo == null || traceNo.equals("")) {

            return recordEntityDao.queryBuilder()
                    .where(TransRecordEntityDao.Properties.TransID.eq(transID.ordinal()),
                            TransRecordEntityDao.Properties.MsgTransDate.eq(date),
                            TransRecordEntityDao.Properties.StatusIccFailTrans.eq(false),
                            TransRecordEntityDao.Properties.StatusNeedReversal.eq(false))
                    .limit(limit).offset(offset)
                    .orderDesc(TransRecordEntityDao.Properties.Id).list();

        } else {
            return recordEntityDao.queryBuilder()
                    .where(TransRecordEntityDao.Properties.MsgTraceNO.like("%" + traceNo + "%"),
                            TransRecordEntityDao.Properties.TransID.eq(transID.ordinal()),
                            TransRecordEntityDao.Properties.MsgTransDate.eq(date),
                            TransRecordEntityDao.Properties.StatusIccFailTrans.eq(false),
                            TransRecordEntityDao.Properties.StatusNeedReversal.eq(false))
                    .limit(limit).offset(offset)
                    .orderDesc(TransRecordEntityDao.Properties.Id).list();
        }
    }

    /**
     * Get the count of records by trace Number(like) and date is today, and specified transaction id
     *
     * @param traceNo trace number
     * @param transID transaction id
     * @return count
     */
    public long searchTraceNoLikeAndTranIdCountToday(String traceNo, @NonNull TransType transID) {
        recordEntityDao.detachAll();
        String date = getCurrentDate();
        if (traceNo == null || traceNo.equals("")) {
            return recordEntityDao.queryBuilder()
                    .where(TransRecordEntityDao.Properties.TransID.eq(transID.ordinal()),
                            TransRecordEntityDao.Properties.MsgTransDate.eq(date),
                            TransRecordEntityDao.Properties.StatusIccFailTrans.eq(false),
                            TransRecordEntityDao.Properties.StatusNeedReversal.eq(false))
                    .count();
        } else {
            return recordEntityDao.queryBuilder()
                    .where(TransRecordEntityDao.Properties.MsgTraceNO.like("%" + traceNo + "%"),
                            TransRecordEntityDao.Properties.TransID.eq(transID.ordinal()),
                            TransRecordEntityDao.Properties.MsgTransDate.eq(date),
                            TransRecordEntityDao.Properties.StatusIccFailTrans.eq(false),
                            TransRecordEntityDao.Properties.StatusNeedReversal.eq(false))
                    .count();

        }

    }

    /**
     * Search records by trace Number(like) and date is before today, and specified transaction id
     *
     * @param traceNo trace number
     * @param transID transaction id
     * @param limit   Max size
     * @param offset  offset
     * @return entities
     */
    public List<TransRecordEntity> searchTraceNoLikeAndTranIdEarlier(String traceNo, TransType transID, int limit, int offset) {
        recordEntityDao.detachAll();
        String sql;
        Cursor cursor;
        String date = getCurrentDate();
        if (traceNo == null) {
            sql = "SELECT * FROM " + TransRecordEntityDao.TABLENAME +
                    " WHERE " + TransRecordEntityDao.Properties.TransID.columnName + "=?" +
                    " AND " + TransRecordEntityDao.Properties.MsgTransDate.columnName + "<?" +
                    " AND " + TransRecordEntityDao.Properties.StatusIccFailTrans.columnName + "=?" +
                    " AND " + TransRecordEntityDao.Properties.StatusNeedReversal.columnName + "=?" +
                    " ORDER BY " + TransRecordEntityDao.Properties.Id.columnName + " DESC" +
                    " LIMIT ? OFFSET ?";
            cursor = recordEntityDao.getDatabase().rawQuery(sql,
                    new String[]{String.valueOf(transID.ordinal()),
                            date,
                            SQL_FALSE,
                            SQL_FALSE,
                            String.valueOf(limit),
                            String.valueOf(offset)
                    });

        } else {
            sql = "SELECT * FROM " + TransRecordEntityDao.TABLENAME +
                    " WHERE " + TransRecordEntityDao.Properties.TransID.columnName + "=?" +
                    " AND " + TransRecordEntityDao.Properties.MsgTraceNO.columnName + " LIKE '%" + traceNo + "%'" +
                    " AND " + TransRecordEntityDao.Properties.MsgTransDate.columnName + "<?" +
                    " AND " + TransRecordEntityDao.Properties.StatusIccFailTrans.columnName + "=?" +
                    " AND " + TransRecordEntityDao.Properties.StatusNeedReversal.columnName + "=?" +
                    " ORDER BY " + TransRecordEntityDao.Properties.Id.columnName + " DESC" +
                    " LIMIT ? OFFSET ?";
            cursor = recordEntityDao.getDatabase().rawQuery(sql,
                    new String[]{String.valueOf(transID.ordinal()),
                            date,
                            SQL_FALSE,
                            SQL_FALSE,
                            String.valueOf(limit),
                            String.valueOf(offset)
                    });
        }
        if ((!cursor.moveToFirst()) || (cursor.getCount() <= 0)) {
            return null;
        }
        List<TransRecordEntity> recordEntities = new ArrayList<>();
        do {
            recordEntities.add(recordEntityDao.readEntity(cursor, 0));
        } while (cursor.moveToNext());

        return recordEntities;
    }

    /**
     * Get the count of records by trace Number(like) and date is before today, and specified transaction id
     *
     * @param traceNo trace number
     * @param transID transaction id
     * @return count
     */
    public long searchTraceNoLikeAndTranIdCountEarlier(String traceNo, @NonNull TransType transID) {
        recordEntityDao.detachAll();
        String sql;
        Cursor cursor;
        String date = getCurrentDate();
        if (traceNo == null || traceNo.equals("")) {
            sql = "SELECT COUNT(*) FROM " + TransRecordEntityDao.TABLENAME +
                    " WHERE " + TransRecordEntityDao.Properties.TransID.columnName + "=?" +
                    " AND " + TransRecordEntityDao.Properties.StatusIccFailTrans.columnName + "=?" +
                    " AND " + TransRecordEntityDao.Properties.StatusNeedReversal.columnName + "=?" +
                    " AND " + TransRecordEntityDao.Properties.MsgTransDate.columnName + "<?";
            cursor = recordEntityDao.getDatabase().rawQuery(sql,
                    new String[]{String.valueOf(transID.ordinal()),
                            SQL_FALSE,
                            SQL_FALSE,
                            date,
                    });
        } else {
            sql = "SELECT COUNT(*) FROM " + TransRecordEntityDao.TABLENAME +
                    " WHERE " + TransRecordEntityDao.Properties.TransID.columnName + "=?" +
                    " AND " + TransRecordEntityDao.Properties.MsgTraceNO.columnName + " LIKE '%" + traceNo + "%'" +
                    " AND " + TransRecordEntityDao.Properties.StatusIccFailTrans.columnName + "=?" +
                    " AND " + TransRecordEntityDao.Properties.StatusNeedReversal.columnName + "=?" +
                    " AND " + TransRecordEntityDao.Properties.MsgTransDate.columnName + "<?";
            cursor = recordEntityDao.getDatabase().rawQuery(sql,
                    new String[]{String.valueOf(transID.ordinal()),
                            SQL_FALSE,
                            SQL_FALSE,
                            date,
                    });
        }
        if ((!cursor.moveToFirst()) || (cursor.getCount() <= 0) || cursor.getString(0) == null) {
            return 0;
        }

        return Integer.parseInt(cursor.getString(0));
    }

    private String getCurrentDate() {
        return new SimpleDateFormat("MMdd", Locale.getDefault()).format(new Date());
    }

    /**
     * getRMBInAmountTotal
     *
     * @return amount
     */
    public String getRMBInAmountTotal() {
        return "0";
    }

    /**
     * getRMBInAmountCount
     *
     * @return amount
     */
    public String getRMBInAmountCount() {
        return "0";
    }

    /**
     * getRMBOutAmountTotal
     *
     * @return amount
     */
    public String getRMBOutAmountTotal() {
        return "0";
    }

    /**
     * getRMBOutAmountCount
     *
     * @return amount
     */
    public String getRMBOutAmountCount() {
        return "0";
    }

    /**
     * get total sale amount (foreign card)
     *
     * @return amount
     */
    public String getForeignInAmountTotal() {
        String sql = "SELECT SUM(" + TransRecordEntityDao.Properties.MsgAmount.columnName + ") FROM " + TransRecordEntityDao.TABLENAME +
                " WHERE (" + TransRecordEntityDao.Properties.TransID.columnName + "=? AND "
                + TransRecordEntityDao.Properties.StatusNeedReversal.columnName + "=? AND "
                + TransRecordEntityDao.Properties.StatusAdjusted.columnName + "=? AND "
                + TransRecordEntityDao.Properties.StatusIccFailTrans.columnName + "=?) " +
                "OR " + TransRecordEntityDao.Properties.TransID.columnName + "=?";
        AppLogUtils.debug(DataOpenHelper.LOG_ENABLE, LOG_TAG, "inAmountTotal sql->" + sql);
        Cursor cursor = recordEntityDao.getDatabase().rawQuery(sql, new String[]{String.valueOf(TransType.SALE.ordinal()),
                SQL_FALSE,
                SQL_FALSE,
                SQL_FALSE,
                String.valueOf(TransType.TIP.ordinal())});
        if ((!cursor.moveToFirst()) || (cursor.getCount() <= 0)) {
            return "0";
        }
        String sum = cursor.getString(0);
        if (sum == null || sum.length() == 0) {
            return "0";
        }
        return sum;
    }

    /**
     * get total sale count(foreign card)
     *
     * @return count
     */
    public String getForeignInAmountCount() {
        String sql = "SELECT COUNT(*) FROM " + TransRecordEntityDao.TABLENAME +
                " WHERE (" + TransRecordEntityDao.Properties.TransID.columnName + "=? AND "
                + TransRecordEntityDao.Properties.StatusNeedReversal.columnName + "=? AND "
                + TransRecordEntityDao.Properties.StatusAdjusted.columnName + "=? AND "
                + TransRecordEntityDao.Properties.StatusIccFailTrans.columnName + "=?) " +
                "OR " + TransRecordEntityDao.Properties.TransID.columnName + "=?";
        Cursor cursor = recordEntityDao.getDatabase().rawQuery(sql, new String[]{String.valueOf(TransType.SALE.ordinal()),
                SQL_FALSE,
                SQL_FALSE,
                SQL_FALSE,
                String.valueOf(TransType.TIP.ordinal())});

        if ((!cursor.moveToFirst()) || (cursor.getCount() <= 0)) {
            return "0";
        }
        return cursor.getString(0);
    }

    /**
     * get total refund amount(foreign card)
     *
     * @return amount
     */
    public String getForeignOutAmountTotal() {
        String sql = "SELECT SUM(" + TransRecordEntityDao.Properties.MsgAmount.columnName + ") FROM " + TransRecordEntityDao.TABLENAME +
                " WHERE (" + TransRecordEntityDao.Properties.TransID.columnName + "=? or "
                + TransRecordEntityDao.Properties.TransID.columnName + "=?) AND "
                + TransRecordEntityDao.Properties.StatusNeedReversal.columnName + "=?";

        AppLogUtils.debug(DataOpenHelper.LOG_ENABLE, LOG_TAG, "outAmountTotal sql->" + sql);

        Cursor cursor = recordEntityDao.getDatabase().rawQuery(sql, new String[]{
                String.valueOf(TransType.VOID.ordinal()),
                String.valueOf(TransType.REFUND.ordinal()),
                SQL_FALSE
        });
        if ((!cursor.moveToFirst()) || (cursor.getCount() <= 0)) {
            return "0";
        }
        // FIXME: 2019/3/29 sync amount in process
        String sum = cursor.getString(0);
        if (sum == null || sum.length() == 0) {
            return "0";
        }

        return cursor.getString(0);
    }

    /**
     * get total refund count(foreign card)
     *
     * @return count
     */
    public String getForeignOutAmountCount() {
        String sql = "SELECT COUNT(*) FROM " + TransRecordEntityDao.TABLENAME +
                " WHERE (" + TransRecordEntityDao.Properties.TransID.columnName + "=? or "
                + TransRecordEntityDao.Properties.TransID.columnName + "=?) AND "
                + TransRecordEntityDao.Properties.StatusNeedReversal.columnName + "=?";

        AppLogUtils.debug(DataOpenHelper.LOG_ENABLE, LOG_TAG, "outAmountTotal sql->" + sql);

        Cursor cursor = recordEntityDao.getDatabase().rawQuery(sql, new String[]{
                String.valueOf(TransType.VOID.ordinal()),
                String.valueOf(TransType.REFUND.ordinal()),
                SQL_FALSE
        });

        if ((!cursor.moveToFirst()) || (cursor.getCount() <= 0)) {
            return "0";
        }

        return cursor.getString(0);
    }

    /**
     * get an IC card Online Success Record
     *
     * @param offset offset
     * @return entity
     */
    public TransRecordEntity getIccOnlineSuccessRecord(int offset) {
        recordEntityDao.detachAll();

        String sql = "SELECT * FROM " + TransRecordEntityDao.TABLENAME +
                " WHERE (" + TransRecordEntityDao.Properties.TransID.columnName + "=? OR " + TransRecordEntityDao.Properties.TransID.columnName + "=?) AND "
                + "(" + TransRecordEntityDao.Properties.CardType.columnName + "=? OR " + TransRecordEntityDao.Properties.CardType.columnName + "=?) AND "
                + TransRecordEntityDao.Properties.StatusIccFailTrans.columnName + "=? AND "
                + TransRecordEntityDao.Properties.StatusOffline.columnName + "=? AND "
                + TransRecordEntityDao.Properties.StatusNeedReversal.columnName + "=? AND "
                + TransRecordEntityDao.Properties.StatusBatchUpload.columnName + "=? LIMIT ? OFFSET ?";

        AppLogUtils.debug(DataOpenHelper.LOG_ENABLE, LOG_TAG, "getIccOnlineSuccessRecord sql->" + sql);
        Cursor cursor = recordEntityDao.getDatabase().rawQuery(sql,
                new String[]{String.valueOf(TransType.SALE.ordinal()),
                        // FIXME: 2018/8/6 void do not need upload tc
                        String.valueOf(TransType.SALE.ordinal()),
                        String.valueOf(CardType.ICC.ordinal()), String.valueOf(CardType.CTLS.ordinal()),
                        SQL_FALSE,
                        SQL_FALSE,
                        SQL_FALSE,
                        String.valueOf(DataConstant.BatchUploadStatus.BATCH_UNFINISH.ordinal()),
                        "1",
                        String.valueOf(offset)
                });
        if ((!cursor.moveToFirst()) || (cursor.getCount() <= 0)) {
            return null;
        }
        return recordEntityDao.readEntity(cursor, 0);
    }

    /**
     * get an IC card Online failed Record
     *
     * @param offset offset
     * @return entity
     */
    public TransRecordEntity getIccOfflineFailRecord(int offset) {
        recordEntityDao.detachAll();

        String sql = "SELECT * FROM " + TransRecordEntityDao.TABLENAME +
                " WHERE " + TransRecordEntityDao.Properties.TransID.columnName + "=? AND"
                + "(" + TransRecordEntityDao.Properties.CardType.columnName + "=? OR " + TransRecordEntityDao.Properties.CardType.columnName + "=?) AND "
                + TransRecordEntityDao.Properties.StatusIccFailTrans.columnName + "=? AND "
                + TransRecordEntityDao.Properties.StatusOffline.columnName + "=? AND "
                + TransRecordEntityDao.Properties.StatusNeedReversal.columnName + "=? AND "
                + TransRecordEntityDao.Properties.StatusBatchUpload.columnName + "=? LIMIT ? OFFSET ?";

//        AppLogUtils.debug("inAmountTotal sql->"+sql);
        Cursor cursor = recordEntityDao.getDatabase().rawQuery(sql,
                new String[]{String.valueOf(TransType.SALE.ordinal()),
                        String.valueOf(CardType.CTLS.ordinal()), String.valueOf(CardType.ICC.ordinal()),
                        SQL_TRUE,
                        SQL_TRUE,
                        SQL_FALSE,
                        String.valueOf(DataConstant.BatchUploadStatus.BATCH_UNFINISH.ordinal()),
                        "1",
                        String.valueOf(offset)
                });
        if ((!cursor.moveToFirst()) || (cursor.getCount() <= 0)) {
            return null;
        }
        return recordEntityDao.readEntity(cursor, 0);
    }

    /**
     * get a Tip Success Record
     *
     * @param offset offset
     * @return entity
     */
    public TransRecordEntity getTipsSuccessRecord(int offset) {
        recordEntityDao.detachAll();

        String sql = "SELECT * FROM " + TransRecordEntityDao.TABLENAME +
                " WHERE " + TransRecordEntityDao.Properties.TransID.columnName + "=? AND "
                + TransRecordEntityDao.Properties.StatusBatchUpload.columnName + "=? AND "
                + TransRecordEntityDao.Properties.StatusNeedReversal.columnName + "=?"
                + " LIMIT ? OFFSET ?";

        Cursor cursor = recordEntityDao.getDatabase().rawQuery(sql,
                new String[]{String.valueOf(TransType.TIP.ordinal()),
                        String.valueOf(DataConstant.BatchUploadStatus.BATCH_UNFINISH.ordinal()),
                        SQL_FALSE,
                        "1", String.valueOf(offset)
                });
        if ((!cursor.moveToFirst()) || (cursor.getCount() <= 0)) {
            return null;
        }
        return recordEntityDao.readEntity(cursor, 0);
    }

    /**
     * get an offline Success Record
     *
     * @param offset offset
     * @return entity
     */
    public TransRecordEntity getOfflineSuccessRecord(int offset) {
        recordEntityDao.detachAll();

        String sql = "SELECT * FROM " + TransRecordEntityDao.TABLENAME +
                " WHERE " + TransRecordEntityDao.Properties.TransID.columnName + "=? AND "
                + "(" + TransRecordEntityDao.Properties.CardType.columnName + "=? OR " + TransRecordEntityDao.Properties.CardType.columnName + "=?) AND "
                + TransRecordEntityDao.Properties.StatusIccFailTrans.columnName + "=? AND "
                + TransRecordEntityDao.Properties.StatusOffline.columnName + "=? AND "
                + TransRecordEntityDao.Properties.StatusNeedReversal.columnName + "=? AND "
                + TransRecordEntityDao.Properties.StatusBatchUpload.columnName + "=? LIMIT ? OFFSET ?";

//        AppLogUtils.debug("inAmountTotal sql->"+sql);
        Cursor cursor = recordEntityDao.getDatabase().rawQuery(sql,
                new String[]{String.valueOf(TransType.SALE.ordinal()),
                        String.valueOf(CardType.ICC.ordinal()), String.valueOf(CardType.CTLS.ordinal()),
                        SQL_FALSE,
                        SQL_TRUE,
                        SQL_FALSE,
                        String.valueOf(DataConstant.BatchUploadStatus.BATCH_UNFINISH.ordinal()),
                        "1",
                        String.valueOf(offset)
                });
        if ((!cursor.moveToFirst()) || (cursor.getCount() <= 0)) {
            return null;
        }
        return recordEntityDao.readEntity(cursor, 0);
    }

    /**
     * get Mag-stripe card Success Records
     *
     * @param offset offset
     * @return entity
     */
    public List<TransRecordEntity> getMagSuccessRecords(int offset, int limit) {
        recordEntityDao.detachAll();

        String sql = "SELECT * FROM " + TransRecordEntityDao.TABLENAME +
                " WHERE (" + TransRecordEntityDao.Properties.TransID.columnName + "=? OR "
                + TransRecordEntityDao.Properties.TransID.columnName + "=?) AND "
                + TransRecordEntityDao.Properties.CardType.columnName + "=? AND "
                + TransRecordEntityDao.Properties.StatusNeedReversal.columnName + "=? AND "
                + TransRecordEntityDao.Properties.StatusBatchUpload.columnName + "=? LIMIT ? OFFSET ?";
//        AppLogUtils.debug(DataOpenHelper.LOG_ENABLE, LOG_TAG, "getMagSuccessRecords sql = {}", sql);

        Cursor cursor = recordEntityDao.getDatabase().rawQuery(sql,
                new String[]{String.valueOf(TransType.SALE.ordinal()),
                        String.valueOf(TransType.VOID.ordinal()),
                        String.valueOf(CardType.MAG.ordinal()),
                        SQL_FALSE,
                        String.valueOf(DataConstant.BatchUploadStatus.BATCH_UNFINISH.ordinal()),
                        String.valueOf(limit),
                        String.valueOf(offset)
                });

        if ((!cursor.moveToFirst()) || (cursor.getCount() <= 0)) {
            return null;
        }
        List<TransRecordEntity> records = new ArrayList<>();
        while (true) {
            records.add(recordEntityDao.readEntity(cursor, 0));
            if (!cursor.moveToNext()) {
                break;
            }
        }
        return records;
    }

    /**
     * get a refund record
     *
     * @param offset offset
     * @return entity
     */
    public TransRecordEntity getRefundRecord(int offset) {
        recordEntityDao.detachAll();

        String sql = "SELECT * FROM " + TransRecordEntityDao.TABLENAME +
                " WHERE " + TransRecordEntityDao.Properties.TransID.columnName + "=? AND "
                + TransRecordEntityDao.Properties.StatusNeedReversal.columnName + "=? AND "
                + TransRecordEntityDao.Properties.StatusBatchUpload.columnName
                + "=? LIMIT ? OFFSET ?";

//        AppLogUtils.debug("inAmountTotal sql->"+sql);
        Cursor cursor = recordEntityDao.getDatabase().rawQuery(sql,
                new String[]{String.valueOf(TransType.REFUND.ordinal()),
                        SQL_FALSE,
                        String.valueOf(DataConstant.BatchUploadStatus.BATCH_UNFINISH.ordinal()),
                        "1",
                        String.valueOf(offset)
                });
        if ((!cursor.moveToFirst()) || (cursor.getCount() <= 0)) {
            return null;
        }
        return recordEntityDao.readEntity(cursor, 0);
    }

    /**
     * get mag-stripe card refund record
     *
     * @param offset offset
     * @return entity
     */
    public TransRecordEntity getMagRefundRecord(int offset) {
        recordEntityDao.detachAll();

        String sql = "SELECT * FROM " + TransRecordEntityDao.TABLENAME +
                " WHERE " + TransRecordEntityDao.Properties.TransID.columnName + "=? AND "
                + TransRecordEntityDao.Properties.CardType.columnName + "=? AND "
                + TransRecordEntityDao.Properties.StatusNeedReversal.columnName + "=? AND "
                + TransRecordEntityDao.Properties.StatusBatchUpload.columnName + "=? LIMIT ? OFFSET ?";

//        AppLogUtils.debug("inAmountTotal sql->"+sql);
        Cursor cursor = recordEntityDao.getDatabase().rawQuery(sql,
                new String[]{String.valueOf(TransType.REFUND.ordinal()),
                        String.valueOf(CardType.MAG.ordinal()),
                        SQL_FALSE,
                        String.valueOf(DataConstant.BatchUploadStatus.BATCH_UNFINISH.ordinal()),
                        "1",
                        String.valueOf(offset)
                });
        if ((!cursor.moveToFirst()) || (cursor.getCount() <= 0)) {
            return null;
        }
        return recordEntityDao.readEntity(cursor, 0);
    }

    /**
     * get IC card refund record
     *
     * @param offset offset
     * @return entity
     */
    public TransRecordEntity getIccRefundRecord(int offset) {
        recordEntityDao.detachAll();

        String sql = "SELECT * FROM " + TransRecordEntityDao.TABLENAME +
                " WHERE " + TransRecordEntityDao.Properties.TransID.columnName + "=? AND "
                + "(" + TransRecordEntityDao.Properties.CardType.columnName + "=? OR "
                + TransRecordEntityDao.Properties.CardType.columnName + "=?) AND "
                + TransRecordEntityDao.Properties.StatusNeedReversal.columnName + "=? AND "
                + TransRecordEntityDao.Properties.StatusBatchUpload.columnName + "=? LIMIT ? OFFSET ?";

//        AppLogUtils.debug("inAmountTotal sql->"+sql);
        Cursor cursor = recordEntityDao.getDatabase().rawQuery(sql,
                new String[]{String.valueOf(TransType.REFUND.ordinal()),
                        String.valueOf(CardType.ICC.ordinal()), String.valueOf(CardType.CTLS.ordinal()),
                        SQL_FALSE,
                        String.valueOf(DataConstant.BatchUploadStatus.BATCH_UNFINISH.ordinal()),
                        "1",
                        String.valueOf(offset)
                });
        if ((!cursor.moveToFirst()) || (cursor.getCount() <= 0)) {
            return null;
        }
        return recordEntityDao.readEntity(cursor, 0);
    }

    /**
     * get IC card sale success record
     *
     * @param offset offset
     * @return entity
     */
    public TransRecordEntity getIccSaleSuccessRecord(int offset) {
        recordEntityDao.detachAll();

        String sql = "SELECT * FROM " + TransRecordEntityDao.TABLENAME +
                " WHERE " + TransRecordEntityDao.Properties.TransID.columnName + "=? AND "
                + "(" + TransRecordEntityDao.Properties.CardType.columnName + "=? OR " + TransRecordEntityDao.Properties.CardType.columnName + "=?) AND "
                + TransRecordEntityDao.Properties.StatusIccFailTrans.columnName + "=? AND "
                + TransRecordEntityDao.Properties.StatusOffline.columnName + "=? AND "
                + TransRecordEntityDao.Properties.StatusNeedReversal.columnName + "=? AND "
                + TransRecordEntityDao.Properties.StatusBatchUpload.columnName + "=? AND "
                + TransRecordEntityDao.Properties.StatusArpcErr.columnName
                + "=? LIMIT ? OFFSET ?";

//        AppLogUtils.debug("inAmountTotal sql->"+sql);
        Cursor cursor = recordEntityDao.getDatabase().rawQuery(sql,
                new String[]{String.valueOf(TransType.SALE.ordinal()),
                        String.valueOf(CardType.ICC.ordinal()), String.valueOf(CardType.CTLS.ordinal()),
                        SQL_FALSE,
                        SQL_FALSE,
                        SQL_FALSE,
                        String.valueOf(DataConstant.BatchUploadStatus.BATCH_UNFINISH.ordinal()),
                        SQL_FALSE,
                        "1",
                        String.valueOf(offset)
                });
        if ((!cursor.moveToFirst()) || (cursor.getCount() <= 0)) {
            return null;
        }
        return recordEntityDao.readEntity(cursor, 0);
    }

    /**
     * get Offline Sale Failed Record
     *
     * @param offset offset
     * @return entity
     */
    public TransRecordEntity getOfflineSaleFailRecord(int offset) {
        recordEntityDao.detachAll();

        String sql = "SELECT * FROM " + TransRecordEntityDao.TABLENAME +
                " WHERE " + TransRecordEntityDao.Properties.TransID.columnName + "=? AND"
                + "(" + TransRecordEntityDao.Properties.CardType.columnName + "=? OR "
                + TransRecordEntityDao.Properties.CardType.columnName + "=?) AND "
                + TransRecordEntityDao.Properties.StatusIccFailTrans.columnName + "=? AND "
                + TransRecordEntityDao.Properties.StatusOffline.columnName + "=? AND "
                + TransRecordEntityDao.Properties.StatusNeedReversal.columnName + "=? AND "
                + TransRecordEntityDao.Properties.StatusBatchUpload.columnName + "=? LIMIT ? OFFSET ?";

//        AppLogUtils.debug("inAmountTotal sql->"+sql);
        Cursor cursor = recordEntityDao.getDatabase().rawQuery(sql,
                new String[]{String.valueOf(TransType.SALE.ordinal()),
                        String.valueOf(CardType.CTLS.ordinal()), String.valueOf(CardType.ICC.ordinal()),
                        SQL_TRUE,
                        SQL_TRUE,
                        SQL_FALSE,
                        String.valueOf(DataConstant.BatchUploadStatus.BATCH_UNFINISH.ordinal()),
                        "1",
                        String.valueOf(offset)
                });
        if ((!cursor.moveToFirst()) || (cursor.getCount() <= 0)) {
            return null;
        }
        return recordEntityDao.readEntity(cursor, 0);
    }

    /**
     * get Offline Transaction Uploading Failed Record
     *
     * @param offset offset
     * @return entity
     */
    public TransRecordEntity getOfflineTransUploadFailedRecord(int offset) {
        recordEntityDao.detachAll();

        String sql = "SELECT * FROM " + TransRecordEntityDao.TABLENAME +
                " WHERE (" + TransRecordEntityDao.Properties.StatusOffineUpload.columnName + "=? OR "
                + TransRecordEntityDao.Properties.StatusOffineUpload.columnName + "=?) AND "
                + TransRecordEntityDao.Properties.StatusNeedReversal.columnName + "=? AND "
                + TransRecordEntityDao.Properties.StatusOffline.columnName + "=? AND "
                + TransRecordEntityDao.Properties.StatusBatchUpload.columnName + "=? "
                + "LIMIT ? "
                + "OFFSET ?";
        AppLogUtils.warn(true, LOG_TAG, "getOfflineTransUploadFailedRecord = {}", sql);
        Cursor cursor = recordEntityDao.getDatabase().rawQuery(sql,
                new String[]{String.valueOf(DataConstant.OffUploadStatus.OFFLINE_DECLINE.ordinal()),
                        String.valueOf(DataConstant.OffUploadStatus.OFFLINE_FAIL.ordinal()),
                        SQL_FALSE,
                        SQL_TRUE,
                        String.valueOf(DataConstant.BatchUploadStatus.BATCH_UNFINISH.ordinal()),
                        "1",
                        String.valueOf(offset)
                });
        if ((!cursor.moveToFirst()) || (cursor.getCount() <= 0)) {
            return null;
        }
        return recordEntityDao.readEntity(cursor, 0);
    }

    /**
     * get ICC APRC Failed Record
     *
     * @param offset offset
     * @return entity
     */
    public TransRecordEntity getIccArpcErrRecord(int offset) {
        recordEntityDao.detachAll();

        String sql = "SELECT * FROM " + TransRecordEntityDao.TABLENAME +
                " WHERE (" + TransRecordEntityDao.Properties.CardType.columnName + "=? OR "
                + TransRecordEntityDao.Properties.CardType.columnName + "=?) AND "
                + TransRecordEntityDao.Properties.TransID.columnName + "=? AND "
                + TransRecordEntityDao.Properties.StatusArpcErr.columnName + "=? AND "
                + TransRecordEntityDao.Properties.StatusNeedReversal.columnName + "=? AND "
                + TransRecordEntityDao.Properties.StatusBatchUpload.columnName + "=? "
                + "LIMIT ? "
                + "OFFSET ?";
        AppLogUtils.warn(DataOpenHelper.LOG_ENABLE, LOG_TAG, "getIccArpcErrRecord = {}", sql);
        Cursor cursor = recordEntityDao.getDatabase().rawQuery(sql,
                new String[]{String.valueOf(CardType.ICC.ordinal()),
                        String.valueOf(CardType.CTLS.ordinal()),
                        String.valueOf(TransType.SALE.ordinal()),
                        SQL_TRUE,
                        SQL_FALSE,
                        String.valueOf(DataConstant.BatchUploadStatus.BATCH_UNFINISH.ordinal()),
                        "1",
                        String.valueOf(offset)
                });
        if ((!cursor.moveToFirst()) || (cursor.getCount() <= 0)) {
            return null;
        }
        return recordEntityDao.readEntity(cursor, 0);
    }

}
