package cn.nexgo.inbas.components.commu.socket;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import org.jetbrains.annotations.NotNull;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.security.KeyManagementException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.Random;
import java.util.concurrent.TimeUnit;

import javax.net.SocketFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactory;

import cn.nexgo.inbas.common.utils.AppUtils;
import cn.nexgo.inbas.components.commu.ICommu;
import cn.nexgo.inbas.components.commu.bean.CommuObject;
import cn.nexgo.inbas.components.commu.socket.factory.ISocket;
import cn.nexgo.inbas.components.commu.socket.factory.SSLSocket;
import cn.nexgo.inbas.components.commu.socket.factory.Socket;
import cn.nexgo.utils.AppFileUtils;
import cn.nexgo.utils.AppLogUtils;
import io.reactivex.Observable;
import io.reactivex.ObservableEmitter;
import io.reactivex.ObservableOnSubscribe;
import io.reactivex.disposables.Disposable;
import io.reactivex.functions.Consumer;
import io.reactivex.schedulers.Schedulers;

/***************************************************************************************************
 *                                  Copyright (C), Nexgo Inc.                                      *
 *                                    http://www.nexgo.cn                                          *
 ***************************************************************************************************
 * usage           : 
 * Version         : 1
 * Author          : Truth
 * Date            : 2017/12/9
 * Modify          : create file
 **************************************************************************************************/
public class SocketTrans implements ICommu {

    private static final CommuObject.ErrorCode PARAMETER_ERROR = new CommuObject.ErrorCode(1, "parameter error");
    private static final CommuObject.ErrorCode FILE_NOT_EXIST = new CommuObject.ErrorCode(2, "file not exist");
    private static final CommuObject.ErrorCode BUSY = new CommuObject.ErrorCode(3, "busy");
    private static final CommuObject.ErrorCode NOT_CONNECTED = new CommuObject.ErrorCode(4, "not connected");
    private static final CommuObject.ErrorCode OTHER_ERROR = new CommuObject.ErrorCode(5, "other error");

    private SocketObject.SocketInitEntity initEntity;
    private ISocket socket;
    @Override
    public void init(@NotNull Object initData, int timeout, @NonNull @NotNull CommuObject.CommonResultListener listener) {
        if(!(initData instanceof SocketObject.SocketInitEntity)){
            listener.onFail(PARAMETER_ERROR);
            return;
        }
        initEntity = (SocketObject.SocketInitEntity)initData;

        if((initEntity.getCerFilePath() != null) && (!AppFileUtils.checkFileExist(initEntity.getCerFilePath()))){  // file not extis
            listener.onFail(FILE_NOT_EXIST);
            return;
        }

        if(initEntity.getCerFilePath() == null){  // socket
            socket = new Socket();
        } else {  // ssl
            socket = new SSLSocket();
//            SocketFactory factory = makeSSLFactory(initEntity.getCerFilePath(), initEntity.getCerFilePwd());
            SocketFactory factory = makeSSLFactory(initEntity.getCerFilePath());
            if(factory == null){
                listener.onFail(PARAMETER_ERROR);
                return;
            }
            ((SSLSocket)socket).setSslFactory(factory);
        }

        listener.onSuccess();
    }

    private SocketFactory makeSSLFactory(String cerfilePath, String pwd){
        SSLContext sslContext = null;
        KeyStore ts = null;
        try {
            ts = KeyStore.getInstance("BKS");
        } catch (KeyStoreException e) {
            e.printStackTrace();
            return null;
        }
        try {
            InputStream is = new FileInputStream(cerfilePath);
            if(pwd == null){
                ts.load(is, null);
            }else{
                ts.load(is, pwd.toCharArray());
            }
        } catch (IOException e) {
            e.printStackTrace();
            return null;
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
            return null;
        } catch (CertificateException e) {
            e.printStackTrace();
            return null;
        }

        TrustManagerFactory tmf = null;
        try {
            tmf = TrustManagerFactory.getInstance("X509");
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        }
        try {
            //noinspection ConstantConditions
            tmf.init(ts);
        } catch (KeyStoreException e) {
            e.printStackTrace();
        }
        TrustManager[] tm = tmf.getTrustManagers();
        try {
            sslContext = SSLContext.getInstance("TLS");
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        }
        try {
            //noinspection ConstantConditions
            sslContext.init(null, tm, null);
        } catch (KeyManagementException e) {
            e.printStackTrace();
        }

        return sslContext.getSocketFactory();
    }


    private SocketFactory makeSSLFactory(String caCertPath) {
        AppLogUtils.debug(true, "soket cerfilePath" ,caCertPath);

        InputStream caInputStream = null;
        Certificate ca = null;
        CertificateFactory cf = null;

        try {
            caInputStream = new FileInputStream(caCertPath);
            AppLogUtils.debug(true, "soket cerfilePath" ,caCertPath);

        } catch (IOException e) {
            e.printStackTrace();
            return null;
        }

        try {
            cf = CertificateFactory.getInstance("X.509");
            ca = cf.generateCertificate(caInputStream);
        } catch (CertificateException e) {
            e.printStackTrace();
        } finally {
            if (caInputStream != null) {
                try {
                    caInputStream.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }

        String keyStoreType = KeyStore.getDefaultType();
        KeyStore keyStore = null;
        try {
            keyStore = KeyStore.getInstance(keyStoreType);
        } catch (KeyStoreException e) {
            e.printStackTrace();
        }

        try {
            keyStore.load(null, null);
        } catch (IOException e) {
            e.printStackTrace();
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        } catch (CertificateException e) {
            e.printStackTrace();
        }

        try {
            keyStore.setCertificateEntry("ca",ca);
        } catch (KeyStoreException e) {
            e.printStackTrace();
        }

        String tmfAlgorithm = "X509";
        TrustManagerFactory tmf = null;
        try {
            tmf = TrustManagerFactory.getInstance(tmfAlgorithm);
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        }

        try {
            tmf.init(keyStore);
        } catch (KeyStoreException e) {
            e.printStackTrace();
        }

        SSLContext context = null;
        try {
            context = SSLContext.getInstance("TLS");
//            context = SSLContext.getInstance("TLSv1.2");
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        }

        try {
            context.init(null, tmf.getTrustManagers(), null);
        } catch (KeyManagementException e) {
            e.printStackTrace();
        }

        return context.getSocketFactory();
    }

    @Nullable
    private CommuObject.StatusListener statusListener;
    private Disposable statusListenDisposable;
    private CommuObject.Status currentStatus = CommuObject.Status.DISCONNECTED;
    @Override
    public void startListenStatus(@NotNull CommuObject.StatusListener listener) {
        this.statusListener = listener;

        if(this.statusListenDisposable != null){  // 正在监听的情况
            return;
        }
        statusListenDisposable = Observable.interval(2, TimeUnit.SECONDS)
                .subscribe(new Consumer<Long>() {
                    @Override
                    public void accept(Long aLong) throws Exception {
                        if(!socket.isConnect()){
                            setConnectStatus(CommuObject.Status.DISCONNECTED);
                            return;
                        }
                    }
                });
    }

    @Override
    public void stopListenStatus() {
        if((statusListenDisposable != null) && (!statusListenDisposable.isDisposed())){  // 取消订阅
            statusListenDisposable.dispose();
        }
        this.statusListener = null;
        this.statusListenDisposable = null;
    }

//    private SocketObject.SocketConnectEntity connectEntity;
    private List<SocketObject.SocketConnectEntity> connectDataList;
    @Override
    public void connect(@NotNull Object connectData, final int timeout, @NonNull @NotNull final CommuObject.CommonResultListener listener) {
        if(!(connectData instanceof List)) {//SocketObject.SocketConnectEntity)){
            listener.onFail(PARAMETER_ERROR);
            return;
        }

        // check Internet statues
        if(!AppUtils.isInternetUseable()){
            listener.onFail(OTHER_ERROR);
            return;
        }

        connectDataList = new ArrayList<>();  // check connect datas
        List<SocketObject.SocketConnectEntity> tmpList = (List<SocketObject.SocketConnectEntity>)connectData;
        for (SocketObject.SocketConnectEntity connectEntity : tmpList) {
            if(connectEntity.isUseable()){
                connectDataList.add(connectEntity);
            }
        }

        if(connectDataList.size() == 0){
            listener.onFail(PARAMETER_ERROR);
            return;
        }

        if(getConnectStatus() == CommuObject.Status.CONNECTED){  // has connected
//            SocketObject.SocketConnectEntity connectEntityTmp = (SocketObject.SocketConnectEntity)connectData;
//
//            if(connectEntityTmp.getAddr().equals(connectEntity.getAddr()) && connectEntityTmp.getPort().equals(connectEntity.getPort())){
//                listener.onSuccess();
//                return;
//            }
            listener.onSuccess();
            return;
        }

        if(getConnectStatus() != CommuObject.Status.DISCONNECTED){
            listener.onFail(BUSY);
            return;
        }

        Observable.create(new ObservableOnSubscribe<Object>() {  // action
            @Override
            public void subscribe(ObservableEmitter<Object> e) throws Exception {
                if(getConnectStatus() == CommuObject.Status.CONNECTED){
                    setConnectStatus(CommuObject.Status.DISCONNECTNIG);
                    socket.close();
                    setConnectStatus(CommuObject.Status.DISCONNECTED);
                }
                setConnectStatus(CommuObject.Status.CONNECTTING);

                while (connectDataList.size() > 0) {
//                    Random random = new Random();
//                    int index = random.nextInt(connectDataList.size());
                    int index = 0;

                    SocketObject.SocketConnectEntity connectEntity = connectDataList.get(index);
                    if(socket.socket(connectEntity.getAddr(), connectEntity.getPort(), timeout)){  //  connect success
                        setConnectStatus(CommuObject.Status.CONNECTED);
                        listener.onSuccess();
                        return;
                    }
                    connectDataList.remove(index);
                }
                setConnectStatus(CommuObject.Status.DISCONNECTED);
                listener.onFail(OTHER_ERROR);

//                if(!socket.socket(connectEntity.getAddr(), connectEntity.getPort(), timeout)){  //  connect fail
//                    setConnectStatus(CommuObject.Status.DISCONNECTED);
//                    listener.onFail(OTHER_ERROR);
//                    return;
//                }
//
//                setConnectStatus(CommuObject.Status.CONNECTED);
//                listener.onSuccess();
            }
        })
                .subscribeOn(Schedulers.io())
                .subscribe();

    }

    @Override
    public void disconnect(int timeout, @NotNull final CommuObject.CommonResultListener listener) {
        if(getConnectStatus() == CommuObject.Status.DISCONNECTED){ // has disconnected
            listener.onSuccess();
            return;
        }
        if(getConnectStatus() != CommuObject.Status.CONNECTED){
            listener.onFail(BUSY);
            return;
        }

        if(statusListener != null){  // status change
            statusListener.onStatus(CommuObject.Status.DISCONNECTNIG);
            return;
        }

        Observable.create(new ObservableOnSubscribe<Object>() {  // action
            @Override
            public void subscribe(ObservableEmitter<Object> e) throws Exception {
                setConnectStatus(CommuObject.Status.DISCONNECTNIG);
                socket.close();
                setConnectStatus(CommuObject.Status.DISCONNECTED);
                listener.onSuccess();
            }
        })
                .subscribeOn(Schedulers.io())
                .subscribe();
    }

    @NonNull
    @Override
    public CommuObject.Status getConnectStatus() {
        if(currentStatus == CommuObject.Status.CONNECTED){
            if(!socket.isConnect()){
                setConnectStatus(CommuObject.Status.DISCONNECTED);
                return CommuObject.Status.DISCONNECTED;
            }
        }

        return currentStatus;
    }

    private void setConnectStatus(CommuObject.Status status){
        if(status == currentStatus){
            return;
        }
        currentStatus = status;
        if(statusListener != null){
            statusListener.onStatus(currentStatus);
        }
    }

    @Override
    public void sendData(@NotNull final byte[] data, final int timeout, @NotNull final CommuObject.CommonResultListener listener) {
        Observable.create(new ObservableOnSubscribe<Object>() {  // action
            @Override
            public void subscribe(ObservableEmitter<Object> e) throws Exception {
                if(getConnectStatus() != CommuObject.Status.CONNECTED){
                    listener.onFail(NOT_CONNECTED);
                    return;
                }

                if(!socket.sendDatas(data, timeout)){   // send fail
                    listener.onFail(OTHER_ERROR);
                    return;
                }

                listener.onSuccess();
            }
        })
                .subscribeOn(Schedulers.io())
                .subscribe();
    }

    private byte[] revcBuf = new byte[1024];
    private RecvObserble<Object> reciveObserble;
    @Override
    public void startRecData(final int timeout, @NotNull final CommuObject.RecvDataListener listener) {
        reciveObserble = new RecvObserble<Object>(timeout, listener);
        Observable.create(reciveObserble).subscribeOn(Schedulers.io()).subscribe();
    }

    private class RecvObserble<T> implements ObservableOnSubscribe<T>{

        private int timeout = 5000;
        private CommuObject.RecvDataListener listener;
        private boolean stopListen = false;

        public RecvObserble(int timeout, CommuObject.RecvDataListener listener){
            this.timeout = timeout;
            this.listener = listener;
        }
        @Override
        public void subscribe(ObservableEmitter<T> e) throws Exception {
            while (true){
                if(getConnectStatus() != CommuObject.Status.CONNECTED){
                    listener.onFail(NOT_CONNECTED);
                    return;
                }

                int readLen = socket.readDatas(revcBuf, 0, revcBuf.length, timeout);
                if(readLen < 0){

                    if(getConnectStatus() == CommuObject.Status.DISCONNECTED){
                        listener.onFail(NOT_CONNECTED);
                        return;
                    }
                    listener.onFail(OTHER_ERROR);
                    return;
                }
                listener.onGetData(Arrays.copyOf(revcBuf, readLen));

                if(stopListen){  // stop loop
                    return;
                }
            }
        }

        public void stopListen(){
            stopListen = true;
        }
    }

    @Override
    public void stopRecData() {
        if(reciveObserble != null){
            reciveObserble.stopListen();
        }
    }

    // **************************************************************************
    private LinkedList<ExeTransEntity> exeTransList = new LinkedList<>();
    private boolean exeTransing = false;
    private int taskID = 0;
    @Override
    public int exchangeMsg(@NotNull Object connectData, @NotNull byte[] sendData, boolean disconnectAfterRecv, @NotNull CommuObject.UnZipDataAction unzipAction, int timeout, @NotNull CommuObject.TransListener listener) {

        ExeTransEntity exeTransConnectData = new ExeTransEntity();

        exeTransConnectData.setConnectData(connectData);
        exeTransConnectData.setSendData(sendData);
        exeTransConnectData.setDisconnectAfterRecv(disconnectAfterRecv);
        exeTransConnectData.setUnzipAction(unzipAction);
        exeTransConnectData.setTimeout(timeout);
        exeTransConnectData.setListener(listener);

        taskID = (taskID+1)%10000;
        exeTransConnectData.setId(taskID);

        exeTransList.add(exeTransConnectData);

        if(!exeTransing){
            exeTransing = true;
            transLoop();
        }

        return taskID;
    }

    @Override
    public void cancelExchangeMsg(int id){
        for (ExeTransEntity exeTransEntity: exeTransList) {
            if(exeTransEntity.getId() == id){
                exeTransEntity.setCancel(true);
                return;
            }
        }
    }

    private static class ExeTransEntity{
        private int id;
        private Object connectData;
        private byte[] sendData;
        private boolean disconnectAfterRecv;
        private CommuObject.UnZipDataAction unzipAction;
        private int timeout;
        private CommuObject.TransListener listener;
        private boolean cancel = false;

        public Object getConnectData() {
            return connectData;
        }

        public void setConnectData(Object connectData) {
            this.connectData = connectData;
        }

        public byte[] getSendData() {
            return sendData;
        }

        public void setSendData(byte[] sendData) {
            this.sendData = sendData;
        }

        public boolean isDisconnectAfterRecv() {
            return disconnectAfterRecv;
        }

        public void setDisconnectAfterRecv(boolean disconnectAfterRecv) {
            this.disconnectAfterRecv = disconnectAfterRecv;
        }

        public CommuObject.UnZipDataAction getUnzipAction() {
            return unzipAction;
        }

        public void setUnzipAction(CommuObject.UnZipDataAction unzipAction) {
            this.unzipAction = unzipAction;
        }

        public int getTimeout() {
            return timeout;
        }

        public void setTimeout(int timeout) {
            this.timeout = timeout;
        }

        public CommuObject.TransListener getListener() {
            return listener;
        }

        public int getId() {
            return id;
        }

        public void setId(int id) {
            this.id = id;
        }

        public void setListener(CommuObject.TransListener listener) {
            this.listener = listener;
        }

        public boolean isCancel() {
            return cancel;
        }

        public void setCancel(boolean cancel) {
            this.cancel = cancel;
        }
    }

    private ExeTransEntity workingExeTransData;
    private boolean socketConnected = false;

    /**
     * exetrans loop
     */
    private void transLoop(){
        if(!exeTransing){
            return;
        }
        if(exeTransList.size() <= 0){
            exeTransing = false;
            return;
        }

        workingExeTransData = exeTransList.getFirst();

        if(workingExeTransData.isCancel()){
            exeTransList.removeFirst();
            workingExeTransData.getListener().onCancel();
            transLoop();
            return;
        }

        connect(workingExeTransData.getConnectData(), workingExeTransData.getTimeout(), exeTransSendListener);
    }

    private CommuObject.CommonResultListener exeTransSendListener = new CommuObject.CommonResultListener() {
        @Override
        public void onSuccess() {
            workingExeTransData.getListener().onConnectSuccess();
            socketConnected = true;
            exeTransSendData();
        }

        @Override
        public void onFail(CommuObject.ErrorCode error) {
            exeTransList.removeFirst();
            workingExeTransData.getListener().onConnectError(error);

            transLoop();
        }
    };

    private void exeTransSendData(){
        if(workingExeTransData.isCancel()){
            exeTransList.removeFirst();
            workingExeTransData.getListener().onCancel();
            transLoop();
            return;
        }

        sendData(workingExeTransData.getSendData(), workingExeTransData.getTimeout(), getExeTransSendListener);
    }

    private CommuObject.CommonResultListener getExeTransSendListener = new CommuObject.CommonResultListener() {
        @Override
        public void onSuccess() {
            workingExeTransData.getListener().onSendSuccess();

            exeTransRecvData();
        }

        @Override
        public void onFail(CommuObject.ErrorCode error) {
            exeTransList.removeFirst();
            workingExeTransData.getListener().onSendError(error);

            transLoop();
        }
    };

    private void exeTransRecvData(){
        if(workingExeTransData.isCancel()){
            exeTransList.removeFirst();
            workingExeTransData.getListener().onCancel();
            transLoop();
            return;
        }

        startRecData(workingExeTransData.getTimeout(), exeTransRecvDataListener);
    }

    private byte[] recvData;
    private CommuObject.RecvDataListener exeTransRecvDataListener = new CommuObject.RecvDataListener() {
        @Override
        public void onGetData(byte[] data) {

            recvData = workingExeTransData.getUnzipAction().getRealData(data);

            if(workingExeTransData.getUnzipAction().getErrorCode() != null){  // data error
                stopRecData();
                exeTransList.removeFirst();
                workingExeTransData.getListener().onReceiveError(workingExeTransData.getUnzipAction().getErrorCode());
                transLoop();
                return;
            }

            if(recvData == null){  // unfinish
                return;
            }

            stopRecData();

            exeTransDisConnect();
        }

        @Override
        public void onFail(CommuObject.ErrorCode error) {
            exeTransList.removeFirst();
            workingExeTransData.getListener().onReceiveError(error);

            transLoop();
        }
    };

    private void exeTransDisConnect(){
        if(!workingExeTransData.isDisconnectAfterRecv()){
            exeTransList.removeFirst();
            workingExeTransData.getListener().onReceiveSuccess(recvData);
            transLoop();
            return;
        }

        if(workingExeTransData.isCancel()){
            exeTransList.removeFirst();
            workingExeTransData.getListener().onCancel();
            transLoop();
            return;
        }

        disconnect(workingExeTransData.getTimeout(), disConnectListener);
    }

    private CommuObject.CommonResultListener disConnectListener = new CommuObject.CommonResultListener() {
        @Override
        public void onSuccess() {
            exeTransList.removeFirst();
            workingExeTransData.getListener().onReceiveSuccess(recvData);
            transLoop();
        }

        @Override
        public void onFail(CommuObject.ErrorCode error) {
            exeTransList.removeFirst();
            workingExeTransData.getListener().onReceiveSuccess(recvData);
            transLoop();
        }
    };

    @Override
    public void enDebug(boolean enable){

    }
}
