package cn.nexgo.inbas.components.commu.serialport;

import android.support.annotation.NonNull;

import com.nexgo.common.ByteUtils;
import com.nexgo.common.LogUtils;
import com.nexgo.oaf.apiv3.APIProxy;
import com.nexgo.oaf.apiv3.SdkResult;
import com.nexgo.oaf.apiv3.device.serialport.SerialCfgEntity;
import com.nexgo.oaf.apiv3.device.serialport.SerialPortDriver;

import org.jetbrains.annotations.NotNull;

import java.util.Arrays;

import cn.nexgo.inbas.components.commu.ICommu;
import cn.nexgo.inbas.components.commu.bean.CommuObject;
import cn.nexgo.utils.BaseUtils;
import io.reactivex.Observable;
import io.reactivex.ObservableEmitter;
import io.reactivex.ObservableOnSubscribe;
import io.reactivex.schedulers.Schedulers;

/***************************************************************************************************
 *                                  Copyright (C), Nexgo Inc.                                      *
 *                                    http://www.nexgo.cn                                          *
 ***************************************************************************************************
 * File Name     : SettleInfoEntity.java
 * Usage         :
 * Version       : 1
 * Author        : lee
 * Date          : 2018/3/19
 * Modification  : Created file

 **************************************************************************************************/

public class SerialPortTrans implements ICommu {

    private static final CommuObject.ErrorCode PARAMETER_ERROR = new CommuObject.ErrorCode(1, "parameter error");
    private static final CommuObject.ErrorCode PORT_NOT_OPEN = new CommuObject.ErrorCode(2, "serialPort not open");
    private static final CommuObject.ErrorCode FAIL = new CommuObject.ErrorCode(3, "Fail");
    private static final CommuObject.ErrorCode NOT_CONNECTED = new CommuObject.ErrorCode(4, "not connected");
    private static final CommuObject.ErrorCode OTHER_ERROR = new CommuObject.ErrorCode(5, "other error");

    private SerialPortDriver portDriver;
    private CommuObject.Status currentStatus = CommuObject.Status.DISCONNECTED; //默认为断开状态

    public SerialPortTrans() {
        portDriver = APIProxy.getDeviceEngine(BaseUtils.getApp().getApplicationContext()).getSerialPortDriver(0);
    }

    @Override
    public void init(Object initData, int timeout, CommuObject.CommonResultListener listener) {
        portDriver.disconnect();
    }

    @Override
    public void startListenStatus(CommuObject.StatusListener listener) {
        listener.onStatus(currentStatus);
    }

    @Override
    public void stopListenStatus() {

    }

    @Override
    public void connect(Object connectData, int timeout, CommuObject.CommonResultListener listener) {
        if (!(connectData instanceof SerialCfgEntity)) {
            listener.onFail(PARAMETER_ERROR);
            return;
        }

        int ret = portDriver.connect((SerialCfgEntity) connectData);
        convertResult(ret, listener);
    }

    @Override
    public void disconnect(int timeout, CommuObject.CommonResultListener listener) {
        int ret = portDriver.disconnect();
        convertResult(ret, listener);
    }

    @NonNull
    @Override
    public CommuObject.Status getConnectStatus() {
        return currentStatus;
    }

    @Override
    public void sendData(final byte[] data, final int timeout, final CommuObject.CommonResultListener listener) {
        LogUtils.debug("发送的数据：{}", ByteUtils.byteArray2HexString(data));
        Observable.create(new ObservableOnSubscribe<Object>() {  // action
            @Override
            public void subscribe(ObservableEmitter<Object> e) throws Exception {
                if (getConnectStatus() != CommuObject.Status.CONNECTED) {
                    listener.onFail(NOT_CONNECTED);
                    return;
                }

                int ret = portDriver.send(data, data.length);
                convertResult(ret, listener);
            }
        })
                .subscribeOn(Schedulers.io())
                .subscribe();
    }

    private void convertResult(int ret, CommuObject.CommonResultListener listener) {
        LogUtils.debug("ret : {}", ret);
        currentStatus = CommuObject.Status.DISCONNECTED;
        switch (ret) {
            case SdkResult.Success:
                listener.onSuccess();
                currentStatus = CommuObject.Status.CONNECTED;
                break;
            case SdkResult.SerialPort_Invalid_Communication_Parameter:
                listener.onFail(PARAMETER_ERROR);
                break;
            case SdkResult.SerialPort_Connect_Fail:
            case SdkResult.SerialPort_Send_Fail:
                listener.onFail(FAIL);
                break;
            case SdkResult.SerialPort_Port_Not_Open:
                listener.onFail(PORT_NOT_OPEN);
                break;
            default:
                listener.onFail(NOT_CONNECTED);
                break;
        }
    }

    /**********************************************************************************************/
    private byte[] recvBuf = new byte[1024];
    private ReceiveObservable<Object> receiveObservable;

    @Override
    public void startRecData(final int timeout, @NotNull final CommuObject.RecvDataListener listener) {
        receiveObservable = new ReceiveObservable<>(timeout, listener);
        Observable.create(receiveObservable).subscribeOn(Schedulers.io()).subscribe();
    }

    private class ReceiveObservable<T> implements ObservableOnSubscribe<T> {

        private int timeout = 5000;
        private CommuObject.RecvDataListener listener;
        private boolean stopListen = false;

        private ReceiveObservable(int timeout, CommuObject.RecvDataListener listener) {
            this.timeout = timeout;
            this.listener = listener;
        }

        @Override
        public void subscribe(ObservableEmitter<T> e) throws Exception {
            while (true) {
                if (getConnectStatus() != CommuObject.Status.CONNECTED) {
                    listener.onFail(NOT_CONNECTED);
                    return;
                }

                int readLen = portDriver.recv(recvBuf, recvBuf.length, timeout);
                if (readLen > 0) {
                    listener.onGetData(Arrays.copyOf(recvBuf, readLen));
                    portDriver.clrBuffer();
                } else {
                    listener.onFail(NOT_CONNECTED);
                    return;
                }

                if (stopListen) {  // stop loop
                    return;
                }
            }
        }

        private void stopListen() {
            stopListen = true;
        }
    }

    @Override
    public void stopRecData() {
        if (receiveObservable != null) {
            receiveObservable.stopListen();
        }
    }

    /**********************************************************************************************/

    @Override
    public int exchangeMsg(final Object connectData, final byte[] sendData, boolean disconnectAfterRecv, CommuObject.UnZipDataAction unzipAction, final int timeout, final CommuObject.TransListener listener) {
        if (!(connectData instanceof SerialCfgEntity)) {
            listener.onConnectError(PARAMETER_ERROR);
            return -1;
        }
        Observable.create(new ObservableOnSubscribe<Object>() {  // action
            int ret = -1;
            @Override
            public void subscribe(ObservableEmitter<Object> e) throws Exception {
                if (getConnectStatus() != CommuObject.Status.CONNECTED) {
                    ret = portDriver.connect((SerialCfgEntity) connectData);
                    if (ret != SdkResult.Success) {
                        currentStatus = CommuObject.Status.DISCONNECTED;
                        portDriver.disconnect();
                        listener.onConnectError(FAIL);
                        return;
                    } else {
                        listener.onConnectSuccess();
                    }
                    currentStatus = CommuObject.Status.CONNECTED;
                }
                ret = portDriver.send(sendData, sendData.length);
                if (ret != SdkResult.Success) {
                    listener.onSendError(FAIL);
                    return;
                } else {
                    listener.onSendSuccess();
                }
                ret = portDriver.recv(recvBuf, recvBuf.length, timeout);
                if (ret != SdkResult.Success) {
                    listener.onReceiveError(FAIL);
                    currentStatus = CommuObject.Status.DISCONNECTED;
                } else {
                    listener.onReceiveSuccess(Arrays.copyOf(recvBuf, ret));
                    portDriver.clrBuffer();
                }

            }
        })
                .subscribeOn(Schedulers.io())
                .subscribe();

        return 0;
    }

    @Override
    public void cancelExchangeMsg(int id) {
        portDriver.disconnect();
        currentStatus = CommuObject.Status.DISCONNECTED;
    }

    @Override
    public void enDebug(boolean enable) {

    }


}
