import type { TEventName, THandler, TListener, TEventMeta, TModuleId } from './types.ts';

/**
 * The EventBus class provides a simple implementation of an event system
 * that allows for registering, unregistering, and emitting events with payloads.
 * Event listeners can be registered to specific event names and triggered accordingly.
 *
 * This approach is based on the **Observer design pattern**, where a single event is propagated to multiple target destinations.
 *
 * The main reason for using this pattern is the requirement to maintain a single instance of the Core, which holds
 * the registered event handlers (e.g. onPaid). These events must be broadcast to all components that are dynamically
 * created and destroyed over time, such as those responsible for managing Apple Pay and Google Pay buttons.
 */
export class EventBus {
    private listeners: Map<TEventName, TListener[]> = new Map();

    /**
     * Register listener
     * @param event
     * @param handler
     */
    register(event: TEventName, handler: THandler): () => void {
        const list = this.listeners.get(event) ?? [];
        list.push({ handler });
        this.listeners.set(event, list);

        // convenient unsubscription
        return () => this.unregister(event, handler);
    }

    /**
     * Unregister listener by key, handler and filterKey
     * @param eventName
     * @param handler
     */
    unregister(eventName: TEventName, handler?: THandler): number {
        const list = this.listeners.get(eventName);
        if (!list) return 0;

        const before = list.length;
        const remain = list.filter((l) => {
            if (handler && l.handler !== handler) return true; // keep
            // remove
            return false;
        });

        if (remain.length > 0) {
            this.listeners.set(eventName, remain);
        } else {
            this.listeners.delete(eventName);
        }

        return before - remain.length;
    }

    /**
     * Emit event
     * @param eventName
     * @param payload
     * @param moduleId
     */
    emit(eventName: TEventName, payload: unknown, moduleId?: TModuleId): void {
        const list = this.listeners.get(eventName);
        if (!list) return;

        const meta: TEventMeta = { eventName, timestamp: Date.now() };

        if (moduleId) {
            meta.moduleId = moduleId;
        }

        for (const l of [...list]) {
            l.handler(payload, meta);
        }
    }

    // /**
    //  * Unregister all listeners with the same key
    //  * @param eventName
    //  */
    // unregisterAll(eventName: TEventName): boolean {
    //     if (!this.listeners.has(eventName)) {
    //         return false;
    //     }
    //     this.listeners.delete(eventName);
    //     return true;
    // }
    //
    // /**
    //  * Unregister all listeners with the same key
    //  * @param eventName
    //  */
    // hasEvent(eventName: TEventName): boolean {
    //     return this.listeners.has(eventName);
    // }
    //
    // /**
    //  * Number of listeners for one event name
    //  * @param eventName
    //  */
    // getEventListenerCount(eventName: TEventName): number {
    //     const array = this.listeners.get(eventName);
    //     if (!array) return 0;
    //     return array.length;
    // }
    //
    // /**
    //  * Number of all unique event names
    //  */
    // getEventKeyCount(): number {
    //     return this.listeners.size;
    // }
}

export const eventBusInstance = new EventBus();
