import { APPLE_PAY_UI, CHECKOUT_ID, GOOGLE_PAY_UI, CHECKOUT_DEBUG, CHECKOUT_LOCALE } from './configuration';
import type { Dispatch, RefObject, SetStateAction } from 'react';
import { eventBusInstance } from '../../services/eventBus';
import { loadComgateCheckout } from './loader';
import type { TComgateCheckoutConstructParameters, TUseComgateCheckoutParams } from './types.ts';

/**
 * Represents the Comgate Checkout class, which encapsulates functionality for initializing,
 * managing, and interacting with payment methods such as Apple Pay and Google Pay.
 */
export class ComgateCheckout {
    private coreInstance: unknown = null;
    private applePayInstance: unknown = null;
    private googlePayInstance: unknown = null;

    /**
     * A state update function for setting the Apple Pay and Google Pay module identifier.
     */
    private setApplePayModuleId: Dispatch<SetStateAction<string | null>>;
    private setGooglePayModuleId: Dispatch<SetStateAction<string | null>>;

    constructor({ setApplePayModuleId, setGooglePayModuleId }: TComgateCheckoutConstructParameters) {
        this.setApplePayModuleId = setApplePayModuleId;
        this.setGooglePayModuleId = setGooglePayModuleId;
    }

    /**
     * Initializes the checkout process, including the loading of required modules
     * (Comgate, Apple Pay, Google Pay) and setting up instances for payment functionalities.
     *
     * @param {RefObject<HTMLDivElement | null>} options.applePayButtonBoxRef - A reference to the HTML div element where the Apple Pay button will be mounted.
     * @param {RefObject<HTMLDivElement | null>} options.googlePayButtonBoxRef - A reference to the HTML div element where the Google Pay button will be mounted.
     * @param {string} options.paymentId - The unique identifier for the payment transaction.
     * @return {Promise<void>} A Promise that resolves when the checkout initialization is complete.
     * @throws {Error} Throws an error if the checkout module or core instance fails to load or initialize.
     */
    public async initCheckout({ applePayButtonBoxRef, googlePayButtonBoxRef, paymentId }: TUseComgateCheckoutParams): Promise<void> {
        let coreInstance = this.getCoreInstance();
        const loadedCheckout = await loadComgateCheckout();

        if (loadedCheckout?.success !== true) {
            throw new Error('Failed to load checkout', {
                cause: {
                    loadedCheckout,
                },
            });
        }

        // Fail-safe in case the instance has already been created on another thread
        coreInstance = this.getCoreInstance();

        // Most likely the first attempt to load the CORE instance, so it needs to be created
        if (!coreInstance) {
            coreInstance = await this.createCoreInstance(loadedCheckout.core, paymentId);
            this.coreInstance = coreInstance;
        }

        // At this point, the instance should already be created, and if not, an error has occurred
        if (!coreInstance) {
            throw new Error('Failed to create CORE instance.');
        }

        // Create Apple Pay instance
        // If the instance cannot be created, an error is thrown and caught during the initCheckout call
        const applePayInstance = await this.createApplePayInstance(loadedCheckout.applepay, coreInstance);

        if (applePayInstance) {
            // Sync Apple Pay module ID for future checks
            this.setApplePayModuleId(applePayInstance.getModuleId() ?? null);
            this.applePayInstance = applePayInstance;
            const applePayMountResult = await this.mountApplePay(applePayInstance, applePayButtonBoxRef);
            if (applePayMountResult) {
                console.log(`Comgate Checkout: The Apple Pay button is ready for the payer.`);
            }
        }

        // Create Google Pay instance
        // If the instance cannot be created, an error is thrown and caught during the initCheckout call
        const googlePayInstance = await this.createGooglePayInstance(loadedCheckout.googlepay, coreInstance);
        if (googlePayInstance) {
            // Sync Google Pay module ID for future checks
            this.setGooglePayModuleId(googlePayInstance.getModuleId() ?? null);
            this.googlePayInstance = googlePayInstance;
            const googlePayMountResult = await this.mountGooglePay(googlePayInstance, googlePayButtonBoxRef);
            if (googlePayMountResult) {
                console.log(`Comgate Checkout: The Google Pay button is ready for the payer.`);
            }
        }
    }

    /**
     * Cleans up and destroys any existing payment instances such as Apple Pay and Google Pay.
     * Releases associated resources and ensures these instances are no longer usable
     * after this method is called.
     */
    public destroy() {
        // console.log('Destroying Apple Pay and Google Pay instances');
        if (this.applePayInstance) {
            this.applePayInstance.destroy();
            this.applePayInstance = null;
            this.setGooglePayModuleId(null);
            // console.log('Apple Pay instance destroyed');
        }
        if (this.googlePayInstance) {
            this.googlePayInstance.destroy();
            this.googlePayInstance = null;
            this.setGooglePayModuleId(null);
            // console.log('Google Pay instance destroyed');
        }
    }

    /**
     * Retrieves the core instance of the checkout. If the local `coreInstance` is defined, it will return that;
     * otherwise, it attempts to access the core instance from the global `window` object.
     *
     * !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
     * !!  Note that on a website, once the Core instance is created, only one instance exists in singleton mode,  !!
     * !!  and it cannot be terminated or recreated at this time.                                                  !!
     * !!  On every second and subsequent call to core.create(), the first instance is always returned.            !!
     * !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
     *
     * @return {Object|null} The core instance of the checkout, or null if not available.
     */
    public getCoreInstance(): object | null {
        return this.coreInstance || window?.comgate?.checkout?.core?.instance;
    }

    /**
     * Asynchronously creates an Apple Pay instance.
     *
     * @param {unknown} applePayInitiators - The object containing methods to initialise Apple Pay from loader.
     * @param {unknown} coreInstance - The core instance required for Apple Pay configuration.
     * @return {Promise<any>} Returns a promise that resolves to the created Apple Pay instance.
     */
    async createApplePayInstance(applePayInitiators: unknown, coreInstance: unknown): Promise<any> {
        return await applePayInitiators.create(coreInstance, {
            ui: APPLE_PAY_UI,
            actions: {
                onButtonClick: (moduleId: string) => {
                    /**
                     * This approach is based on the **Observer design pattern**, where a single event is propagated to multiple target destinations.
                     *
                     * The main reason for using this pattern is the requirement to maintain a single instance of the Core, which holds
                     * the registered event handlers (e.g. onPaid). These events must be broadcast to all components that are dynamically
                     * created and destroyed over time, such as those responsible for managing Apple Pay and Google Pay buttons.
                     */
                    eventBusInstance.emit('apple-pay-button-clicked', null, moduleId);
                },
            },
        });
    }

    /**
     * Mounts the Apple Pay button onto the specified DOM element if Apple Pay is supported and can make payments.
     *
     * @param {unknown} applePayInstance An instance of Apple Pay.
     * @param {RefObject<HTMLDivElement | null>} selector A reference to the DOM element where the Apple Pay button will be mounted.
     * @return {Promise<boolean>} A promise that resolves to `true` if the Apple Pay button is mounted successfully, or `false` if not.
     */
    async mountApplePay(applePayInstance: any, selector: RefObject<HTMLDivElement | null>): Promise<boolean> {
        const can = await applePayInstance.canMakePayments().catch(() => {
            console.warn('Comgate Checkout: Apple Pay canMakePayments() returned false.');
            return false;
        });

        if (!can) {
            return false;
        }

        if (!selector.current) {
            console.warn(`Comgate Checkout: Apple Pay mount target element was not found.`);
            return false;
        }

        return await applePayInstance
            .mount([selector.current])
            // then is not needed as it would only do `return true`
            // .then((result) => { return true; }) // result is always true in then
            .catch((error) => {
                console.error('Comgate Checkout: Error mounting Apple Pay button.', error);
                return false;
            });
    }

    /**
     * Asynchronously creates an Google Pay instance.
     *
     * @param {unknown} googlePayInitiators The object containing methods to initialise Google Pay from loader.
     * @param {unknown} coreInstance The core instance required for Google Pay configuration.
     * @return {Promise<any>} Returns a promise that resolves to the created Google Pay instance.
     */
    async createGooglePayInstance(googlePayInitiators: unknown, coreInstance: unknown): Promise<any> {
        return await googlePayInitiators.create(coreInstance, {
            ui: GOOGLE_PAY_UI,
            actions: {
                onButtonClick: (moduleId: string) => {
                    /**
                     * This approach is based on the **Observer design pattern**, where a single event is propagated to multiple target destinations.
                     *
                     * The main reason for using this pattern is the requirement to maintain a single instance of the Core, which holds
                     * the registered event handlers (e.g. onPaid). These events must be broadcast to all components that are dynamically
                     * created and destroyed over time, such as those responsible for managing Apple Pay and Google Pay buttons.
                     */
                    eventBusInstance.emit('google-pay-button-clicked', null, moduleId);
                },
            },
        });
    }

    /**
     * Mounts the Google Pay button onto the specified DOM element if Google Pay is supported and can make payments.
     *
     * @param {unknown} googlePayInstance An instance of Google Pay.
     * @param {RefObject<HTMLDivElement | null>} selector A reference to the DOM element where the Google Pay button will be mounted.
     * @return {Promise<boolean>} A promise that resolves to `true` if the Google Pay button is mounted successfully, or `false` if not.
     */
    async mountGooglePay(googlePayInstance: unknown, selector: RefObject<HTMLDivElement | null>): Promise<boolean> {
        const can = await googlePayInstance.canMakePayments().catch(() => {
            console.warn('Comgate Checkout: Google Pay canMakePayments() returned false.');
            return false;
        });

        if (!can) {
            return false;
        }

        if (!selector.current) {
            console.warn(`Comgate Checkout: Google Pay mount target element was not found.`);
            return false;
        }

        return await googlePayInstance
            .mount([selector.current])
            // then is not needed as it would only do `return true`
            // .then((result) => { return true; }) // result is always true in then
            .catch((error) => {
                console.error('Comgate Checkout: Error mounting Google Pay button.', error);
                return false;
            });
    }

    /**
     * Creates and initializes a core instance using the provided initiators and configuration details.
     *
     * @param {unknown} coreInitiators - The object responsible for initiating the core instance creation process.
     * @param {string} paymentId - The payment identifier associated with the transaction.
     * @return {Promise<unknown>} A Promise resolving to the created core instance.
     */
    async createCoreInstance(coreInitiators: unknown, paymentId: string) {
        // console.log('coreInitiators', coreInitiators);

        /**
         * @note eventBusInstance.emit(...)
         * This approach is based on the **Observer design pattern**, where a single event is propagated to multiple target destinations.
         *
         * The main reason for using this pattern is the requirement to maintain a single instance of the Core, which holds
         * the registered event handlers (e.g. onPaid). These events must be broadcast to all components that are dynamically
         * created and destroyed over time, such as those responsible for managing Apple Pay and Google Pay buttons.
         */
        return await coreInitiators.create({
            checkoutId: CHECKOUT_ID,
            transactionId: paymentId,
            locale: CHECKOUT_LOCALE,
            debug: CHECKOUT_DEBUG,
            onPaid: (p: unknown, moduleId?: string) => {
                // console.log('onPaid', moduleId, p);
                eventBusInstance.emit('checkout-paid', p, moduleId);
            },
            onCancelled: (p: unknown, moduleId?: string) => {
                // console.log('onCancelled', moduleId, p);
                eventBusInstance.emit('checkout-cancelled', p, moduleId);
            },
            onPending: (p: unknown, moduleId?: string) => {
                // console.log('onPending', moduleId, p);
                eventBusInstance.emit('checkout-pending', p, moduleId);
            },
            onError: (p: unknown, moduleId?: string) => {
                // console.log('onError', moduleId, p);
                eventBusInstance.emit('checkout-error', p, moduleId);
            },
            onPaymentStarted: (moduleId?: string) => {
                // console.log('onPaymentStarted', moduleId)
                eventBusInstance.emit('checkout-payment-started', null, moduleId);
            },
            onPaymentStopped: (moduleId?: string) => {
                // console.log('onPaymentStopped', moduleId);
                eventBusInstance.emit('checkout-payment-started', null, moduleId);
            },
        });
    }
}
