import { useEffect, useMemo, useRef, useState } from 'react';
import './styles.less';
import { ComgateCheckout, type TStatus } from '../../shared/comgate/checkout';
import { eventBusInstance, type TEventMeta } from '../../shared/services/eventBus';

/**
 * A React functional component that initializes and manages the Comgate Checkout SDK,
 * handling events and rendering payment buttons and relevant status messages.
 *
 * @param {string} props.paymentId - The unique identifier for the payment being processed.
 */
export default function PayButtonsContainer({ paymentId, onFinal }: { paymentId: string; onFinal: () => void }) {
    const isFirstRender = useRef(true);

    /**
     * The value indicates the progress of the checkout creation and payment process.
     */
    const [status, setStatus] = useState<TStatus>('idle');

    /**
     * Checkout module IDs
     */
    const [applePayModuleId, setApplePayModuleId] = useState<string | null>(null);
    const [googlePayModuleId, setGooglePayModuleId] = useState<string | null>(null);

    /**
     * A React ref that holds an array of unsubscribe functions for registered listeners.
     */
    const registeredListenersUnsubscribeRef = useRef<Array<() => void>>([]);

    /**
     *  Ref to elements for displaying Comgate Checkout buttons
     */
    const applePayButtonBoxRef = useRef<HTMLDivElement>(null);
    const googlePayButtonBoxRef = useRef<HTMLDivElement>(null);

    /**
     * A React ref object used to store module identifiers related to payment systems.
     * The object tracks module IDs for Apple Pay and Google Pay.
     */
    const moduleIdsRef = useRef<{ applePayModuleId: string | null; googlePayModuleId: string | null }>({
        applePayModuleId: null,
        googlePayModuleId: null,
    });

    useEffect(() => {
        if (status == 'payment-paid' || status == 'payment-cancelled') {
            onFinal();
        }
    }, [status]);

    /**
     * Reactively update applePayModuleId
     */
    useEffect(() => {
        moduleIdsRef.current.applePayModuleId = applePayModuleId;
    }, [applePayModuleId]);

    /**
     * Reactively update googlePayModuleId
     */
    useEffect(() => {
        moduleIdsRef.current.googlePayModuleId = googlePayModuleId;
    }, [googlePayModuleId]);

    /**
     * Represents a memoized instance of the ComgateCheckout service used for managing the checkout process.
     * The service is initialized with provided callbacks for handling Apple Pay and Google Pay module IDs.
     *
     * Dependencies:
     * - `setApplePayModuleId`: A callback function to set the module ID for Apple Pay integration.
     * - `setGooglePayModuleId`: A callback function to set the module ID for Google Pay integration.
     *
     * This variable leverages React’s useMemo to ensure that the ComgateCheckout instance is not re-created during the component’s lifecycle.
     */
    const checkoutService = useMemo(() => new ComgateCheckout({ setApplePayModuleId, setGooglePayModuleId }), []);

    /**
     * Intelligently initialises the Core and module instances of the Comgate Checkout SDK when the component mounts.
     */
    useEffect(() => {
        onCreate();
        // on destroy
        return () => {
            onDestroy();
        };
    }, []);

    /**
     * onCreate initialises the checkout process and registers the necessary event listeners
     * only once at the beginning of the component’s lifecycle.
     */
    const onCreate = () => {
        registerEventListeners();
        checkoutService
            .initCheckout({ applePayButtonBoxRef, googlePayButtonBoxRef, paymentId })
            .then(() => {
                setStatus('ready');
            })
            .catch((error: unknown) => {
                setStatus('error');
                console.error(error);
            });
    };

    /**
     * onDestroy is executed to clean up resources and unregister event listeners
     * at the end of the component’s lifecycle.
     */
    const onDestroy = () => {
        unregisterEventListeners();
        checkoutService.destroy();
    };

    /**
     * When a payment ID change is requested, it calls `change` on the CORE checkout instance.
     */
    useEffect(() => {
        if (isFirstRender.current) {
            // skips the first call on mount
            isFirstRender.current = false;
            return;
        }

        // console.log('CHANGING PAYMENT ID:', paymentId);
        checkoutService.getCoreInstance()?.changePayment(paymentId);
    }, [paymentId]);

    /**
     * Registers event listeners for various checkout-related events.
     * This method binds specific event handlers to corresponding event types.
     */
    const registerEventListeners = (): void => {
        /**
         * This approach is based on the **Observer design pattern**, where a single event is propagated to multiple target destinations.
         *
         * The main reason for using this pattern is the requirement to maintain a single instance of the Core, which holds
         * the registered event handlers (e.g. onPaid). These events must be broadcast to all components that are dynamically
         * created and destroyed over time, such as those responsible for managing Apple Pay and Google Pay buttons.
         */
        const unsubs = registeredListenersUnsubscribeRef.current;
        unsubs.push(eventBusInstance.register('checkout-paid', onPaidHandler));
        unsubs.push(eventBusInstance.register('checkout-cancelled', onCancelledHandler));
        unsubs.push(eventBusInstance.register('checkout-pending', onPendingHandler));
        unsubs.push(eventBusInstance.register('checkout-error', onErrorHandler));
        unsubs.push(
            eventBusInstance.register('apple-pay-button-clicked', (payload: unknown, meta: TEventMeta) => onButtonClicked('applepay', payload, meta))
        );
        unsubs.push(
            eventBusInstance.register('google-pay-button-clicked', (payload: unknown, meta: TEventMeta) => onButtonClicked('googlepay', payload, meta))
        );
    };

    /**
     * Unregisters all event listeners that have previously been registered.
     * This method assumes that `registeredListeners` is an array of functions,
     * each representing a hook to unregister a specific event listener.
     * It iterates over the array and calls each function to ensure proper cleanup.
     */
    const unregisterEventListeners = (): void => {
        for (const unregister of registeredListenersUnsubscribeRef.current) unregister();
        registeredListenersUnsubscribeRef.current = [];
    };

    /**
     * Handles the event when a payment is successfully made.
     *
     * @param {unknown} p - The data related to the payment event.
     * @param {TEventMeta} meta - Metadata associated with the event.
     * @return {void} Doesn't return anything.
     */
    const onPaidHandler = (p: unknown, meta: TEventMeta) => {
        // Be careful to avoid double PAID handling
        console.log('onPaidHandler', p, meta);
        setStatus('payment-paid');
    };

    /**
     * Handles the event wherein a payment or action is cancelled.
     *
     * @param {unknown} p - The primary parameter that contains data or information related to the cancellation event.
     * @param {TEventMeta} meta - Metadata related to the cancellation event, providing additional context or details.
     */
    const onCancelledHandler = (p: unknown, meta: TEventMeta) => {
        // Be careful to avoid double CANCELLED handling
        console.log('onCancelledHandler', p, meta);
        setStatus('payment-cancelled');
    };

    /**
     * Handles the pending state of a payment process.
     *
     * @param {unknown} p - The parameter representing any dynamic data associated with the event.
     * @param {TEventMeta} meta - Metadata associated with the event, including moduleId that indicates the source of the event.
     */
    const onPendingHandler = (p: unknown, meta: TEventMeta) => {
        console.log('onPendingHandler', p, meta);

        const { applePayModuleId, googlePayModuleId } = moduleIdsRef.current;
        // Do it only when the event is emitted by my instance.
        if (meta.moduleId && [applePayModuleId, googlePayModuleId].includes(meta.moduleId)) {
            console.log('onPendingHandler INSIDE', meta.moduleId);
            setStatus('payment-pending');
        }
    };

    /**
     * Handles errors emitted by specific payment modules and sets the payment status to "payment-error".
     *
     * @param {unknown} p - The error payload or information received during the error event.
     * @param {TEventMeta} meta - Metadata of the event, including the module ID and other event details.
     */
    const onErrorHandler = (p: unknown, meta: TEventMeta) => {
        console.log('onErrorHandler', p, meta);

        // Do it only when the event is emitted by my instance.
        if (meta.moduleId && [applePayModuleId, googlePayModuleId].includes(meta.moduleId)) {
            console.log('onErrorHandler INSIDE', meta.moduleId);
            setStatus('payment-error');
        }
    };

    /**
     * Handles errors emitted by specific payment modules and sets the payment status to "payment-error".
     *
     * @param service - The name of the service associated with the button click event.
     * @param {unknown} p - The error payload or information received during the error event.
     * @param {TEventMeta} meta - Metadata of the event, including the module ID and other event details.
     */
    const onButtonClicked = (service: string, p: unknown, meta: TEventMeta) => {
        console.log('onButtonClicked', service, p, meta);

        // Do it only when the event is emitted by my instance.
        if (meta.moduleId && [googlePayModuleId, applePayModuleId].includes(meta.moduleId)) {
            console.log('onButtonClicked INSIDE', meta.moduleId);
            console.log('onButtonClicked', service, p, meta);
        }
    };

    /**
     * A record of status messages used to communicate checkout and payment statuses.
     */
    const statusMessages: Record<string, string> = {
        'payment-paid': 'The payment has been completed.',
        'payment-pending': 'The attempt was unsuccessful. Check the console for details and try again.',
        'payment-error': 'The payment could not be completed. Check the console for details. ',
        ready: 'Checkout is ready.',
        error: 'Checkout failed to initialise. Check the console for details.',
    };

    /**
     * Render template
     */
    return (
        <div className='page'>
            {!['payment-cancelled', 'payment-paid'].includes(status) && (
                <div className='container'>
                    <div
                        className='button'
                        ref={applePayButtonBoxRef}
                    />
                    <div
                        className='button'
                        ref={googlePayButtonBoxRef}
                    />

                    <div
                        className='unique-id-box'
                        title='Apple Pay module instance ID'
                    >
                        AP instanceId: {applePayModuleId}
                    </div>
                    <div
                        className='unique-id-box'
                        title='Google Pay module instance ID'
                    >
                        GP instanceId: {googlePayModuleId}
                    </div>
                </div>
            )}

            {statusMessages[status] && <p className={`status-message ${status}`}>{statusMessages[status]}</p>}
        </div>
    );
}
