package cn.nexgo.hwdriver;

import android.support.annotation.NonNull;

import com.nexgo.common.Des;
import com.nexgo.oaf.apiv3.APIProxy;
import com.nexgo.oaf.apiv3.SdkResult;
import com.nexgo.oaf.apiv3.device.pinpad.MacAlgorithmModeEnum;
import com.nexgo.oaf.apiv3.device.pinpad.PinPad;
import com.nexgo.oaf.apiv3.device.pinpad.WorkKeyTypeEnum;

import java.util.Arrays;

import cn.nexgo.hwdriver.internal.AppCommonActionListener;
import cn.nexgo.utils.AppLogUtils;
import cn.nexgo.utils.BaseUtils;
import cn.nexgo.utils.ErrorCode;
import cn.nexgo.utils.MixUtils;
import io.reactivex.Observable;
import io.reactivex.ObservableEmitter;
import io.reactivex.ObservableOnSubscribe;
import io.reactivex.schedulers.Schedulers;

/***************************************************************************************************
 *                                  Copyright (C), Nexgo Inc.                                      *
 *                                    http://www.nexgo.cn                                          *
 ***************************************************************************************************
 * usage           : 
 * Version         : 1
 * Author          : Truth
 * Date            : 2018/1/8
 * Modify          : create file
 **************************************************************************************************/
public class AppPinPad {
    private static boolean LOG_ENABLE = true;
    private static String LOG_TAG = "AppPinPad";

    public static final int OTHER_ERRORCODE = -1;
    public static final int PARAMERROR_ERRORCODE = 2;
    public static final int KCVERROR_ERRORCODE = 3;

    private PinPad pinPad;
    public AppPinPad(){
        pinPad = APIProxy.getDeviceEngine(BaseUtils.getApp().getApplicationContext()).getPinPad();
    }

    /**
     * write plain main key
     * @param keyIndex
     * @param keyData
     * @param kcv
     * @param listenner
     */
    public void writeMKPlain(final int keyIndex, final @NonNull byte[] keyData, final byte[] kcv, final @NonNull AppCommonActionListener listenner){
        if((keyIndex >= 20) || (keyIndex < 0)){
            AppLogUtils.error(LOG_ENABLE, LOG_TAG, "AppPinPad writeMKPlain keyIndex error ->"+keyIndex);
            listenner.onFail(new ErrorCode(PARAMERROR_ERRORCODE, "key index error"));
            return;
        }

        if((keyData.length != 8) && (keyData.length != 16) && (keyData.length != 24)){
            AppLogUtils.error(LOG_ENABLE, LOG_TAG, "AppPinPad writeMKPlain keyData len error ->"+keyData.length);
            listenner.onFail(new ErrorCode(PARAMERROR_ERRORCODE, "keyData len error"));
            return;
        }

        if((kcv != null) && (kcv.length != 4)){
            AppLogUtils.error(LOG_ENABLE, LOG_TAG, "AppPinPad writeMKPlain kcv len error ->"+kcv.length);
            listenner.onFail(new ErrorCode(PARAMERROR_ERRORCODE, "kcv len error"));
            return;
        }


        Observable.create(new ObservableOnSubscribe<Object>() {
            @Override
            public void subscribe(ObservableEmitter<Object> e) throws Exception {

                if(kcv != null){  // check kcv
                    byte[] kcvTmpData = new byte[8];
                    Arrays.fill(kcvTmpData, (byte)0x00);
                    byte[] calcKcv;
                    if(keyData.length == 8){
                        calcKcv = Des.des_crypt(keyData, kcvTmpData);
                    }else{
                        calcKcv = Des.trides_crypt(keyData, kcvTmpData);
                    }
                    if(!MixUtils.compare(calcKcv, kcv)){
                        listenner.onFail(new ErrorCode(KCVERROR_ERRORCODE, "key error"));
                        return;
                    }
                }

                int ret = pinPad.writeMKey(keyIndex, keyData, keyData.length);
                if(ret != SdkResult.Success){
                    listenner.onFail(new ErrorCode(OTHER_ERRORCODE, "write fail->"+ret));
                    return;
                }
                listenner.onSuccess();
                e.onComplete();
            }
        })
                .subscribeOn(Schedulers.io())
                .subscribe();

    }

    /**
     * write encrypted main key
     * @param keyIndex
     * @param keyData
     * @param kcv
     * @param decMKIndex
     * @param listenner
     */
    public void writeMK(final int keyIndex, final @NonNull byte[] keyData, final byte[] kcv, final int decMKIndex, final @NonNull AppCommonActionListener listenner){
        if((keyIndex >= 20) || (keyIndex < 0)){
            AppLogUtils.error(LOG_ENABLE, LOG_TAG, "AppPinPad writeMK keyIndex error ->"+keyIndex);
            listenner.onFail(new ErrorCode(PARAMERROR_ERRORCODE, "key index error"));
            return;
        }

        if((keyData.length != 8) && (keyData.length != 16) && (keyData.length != 24)){
            AppLogUtils.error(LOG_ENABLE, LOG_TAG, "AppPinPad writeMK keyData len error ->"+keyData.length);
            listenner.onFail(new ErrorCode(PARAMERROR_ERRORCODE, "keyData len error"));
            return;
        }

        if((decMKIndex >= 20) || (decMKIndex < 0)){
            AppLogUtils.error(LOG_ENABLE, LOG_TAG, "AppPinPad writeMK keyIndex error ->"+keyIndex);
            listenner.onFail(new ErrorCode(PARAMERROR_ERRORCODE, "dec key index error"));
            return;
        }

        Observable.create(new ObservableOnSubscribe<Object>() {
            @Override
            public void subscribe(ObservableEmitter<Object> e) throws Exception {
                if(!pinPad.isKeyExist(keyIndex)){
                    listenner.onFail(new ErrorCode(OTHER_ERRORCODE, "TMK Is Not Exist"));
                    return;
                }
                int ret = pinPad.writeMKey(keyIndex, keyData, keyData.length, decMKIndex);
                if(ret != SdkResult.Success){
                    listenner.onFail(new ErrorCode(OTHER_ERRORCODE, "write fail->"+ret));
                    return;
                }
                listenner.onSuccess();
                e.onComplete();
            }
        })
                .subscribeOn(Schedulers.io())
                .subscribe();
    }

    /**
     * write work key
     * @param wkType
     * @param keyIndex
     * @param keyData
     * @param kcv
     * @param listenner
     */
    public void writeWK(final @NonNull WKType wkType, final int keyIndex, final @NonNull byte[] keyData, final byte[] kcv, final @NonNull AppCommonActionListener listenner){

        if((keyIndex >= 20) || (keyIndex < 0)){
            AppLogUtils.error(LOG_ENABLE, LOG_TAG, "AppPinPad writeWK keyIndex error ->"+keyIndex);
            listenner.onFail(new ErrorCode(PARAMERROR_ERRORCODE, "key index error"));
            return;
        }

        if((keyData.length != 8) && (keyData.length != 16) && (keyData.length != 24)){
            AppLogUtils.error(LOG_ENABLE, LOG_TAG, "AppPinPad writeWK keyData len error ->"+keyData.length);
            listenner.onFail(new ErrorCode(PARAMERROR_ERRORCODE, "keyData len error"));
            return;
        }

        if((kcv != null) && (kcv.length != 4)){
            AppLogUtils.error(LOG_ENABLE, LOG_TAG, "AppPinPad writeWK kcv len error ->"+kcv.length);
            listenner.onFail(new ErrorCode(PARAMERROR_ERRORCODE, "kcv len error"));
            return;
        }

        Observable.create(new ObservableOnSubscribe<Object>() {
            @Override
            public void subscribe(ObservableEmitter<Object> e) throws Exception {
                if(!pinPad.isKeyExist(keyIndex)){
                    listenner.onFail(new ErrorCode(OTHER_ERRORCODE, "TMK Is Not Exist"));
                    return;
                }
                int ret = pinPad.writeWKey(keyIndex, keyTypeChange(wkType), keyData, keyData.length);
                if(ret != SdkResult.Success){
                    listenner.onFail(new ErrorCode(OTHER_ERRORCODE, "write fail->"+ret));
                    return;
                }

                if(kcv != null){  // check kcv
                    byte[] calcKcv = pinPad.calcWKeyKCV(keyIndex, keyTypeChange(wkType));
                    if(!MixUtils.compare(calcKcv, kcv)){
                        listenner.onFail(new ErrorCode(KCVERROR_ERRORCODE, "key error"));
                        return;
                    }
                }

                listenner.onSuccess();
                e.onComplete();
            }
        })
                .subscribeOn(Schedulers.io())
                .subscribe();
    }

    /**
     * calc mac
     * @param keyIndex
     * @param data
     * @param macAlgorithmEnum
     * @param listenner
     */
    public void calcMac(final int keyIndex, final @NonNull byte[] data, final @NonNull MacAlgorithmEnum macAlgorithmEnum,  final @NonNull CalcMacListenner listenner){
        if((keyIndex >= 20) || (keyIndex < 0)){
            AppLogUtils.error(LOG_ENABLE, LOG_TAG, "AppPinPad calcMac keyIndex error ->"+keyIndex);
            listenner.onFail(new ErrorCode(PARAMERROR_ERRORCODE, "key index error"));
            return;
        }
        if(data.length == 0){
            AppLogUtils.error(LOG_ENABLE, LOG_TAG, "AppPinPad calcMac keyIndex error ->"+keyIndex);
            listenner.onFail(new ErrorCode(PARAMERROR_ERRORCODE, "key data len -> 0"));
            return;
        }

        Observable.create(new ObservableOnSubscribe<Object>() {
            @Override
            public void subscribe(ObservableEmitter<Object> e) throws Exception {
                byte[] formatedData = data;
                if(data.length % 8 != 0){
                    formatedData = new byte[(data.length + 7) / 8 * 8];
                    Arrays.fill(formatedData, (byte)0x00);
                    System.arraycopy(data, 0, formatedData, 0, data.length);
                }
                byte[] ret = pinPad.calcMac(keyIndex, macAlgorithmTypeChange(macAlgorithmEnum), formatedData);
                if(ret == null){
                    listenner.onFail(new ErrorCode(OTHER_ERRORCODE, "calc fail"));
                    return;
                }

                listenner.onSuccess(ret);
                e.onComplete();
            }
        })
                .subscribeOn(Schedulers.io())
                .subscribe();
    }

    @NonNull
    private WorkKeyTypeEnum keyTypeChange(@NonNull WKType keyType){
        switch (keyType){
            case MACKEY:
                return WorkKeyTypeEnum.MACKEY;
            case PINKEY:
                return WorkKeyTypeEnum.PINKEY;
            default:
                return WorkKeyTypeEnum.TDKEY;
        }
    }

    public enum WKType{
        PINKEY,MACKEY,TDK
    }

    @NonNull
    private MacAlgorithmModeEnum macAlgorithmTypeChange(@NonNull MacAlgorithmEnum algorithm){
        switch (algorithm){
            case ECB:
                return MacAlgorithmModeEnum.ECB;
            case CBC:
                return MacAlgorithmModeEnum.CBC;
            case X919:
                return MacAlgorithmModeEnum.X919;
            default:
                return MacAlgorithmModeEnum.MAC9606;
        }
    }

    public enum MacAlgorithmEnum {
        ECB,
        CBC,
        X919,
        MAC9606,
    }

    public interface CalcMacListenner{
        void onSuccess(byte[] macData);
        void onFail(ErrorCode errorCode);
    }
}
