package cn.nexgo.inbas.transactions.common.protocol.model;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.List;

import cn.nexgo.hwdriver.AppPinPad;
import cn.nexgo.inbas.R;
import cn.nexgo.inbas.common.GData;
import cn.nexgo.inbas.common.bean.ErrorCode;
import cn.nexgo.inbas.common.utils.SysTime;
import cn.nexgo.inbas.components.commu.ICommu;
import cn.nexgo.inbas.components.commu.bean.CommuObject;
import cn.nexgo.inbas.components.commu.socket.SocketObject;
import cn.nexgo.inbas.transactions.common.protocol.MsgModel;
import cn.nexgo.inbas.transactions.common.protocol.bean.MsgBean;
import cn.nexgo.inbas.transactions.common.protocol.utils.Cn8583Unzip;
import cn.nexgo.protocol.iso8583.ConfigParser;
import cn.nexgo.protocol.iso8583.Message;
import cn.nexgo.protocol.iso8583.MessageFactory;
import cn.nexgo.utils.BaseUtils;
import cn.nexgo.utils.ConvertUtils;
import cn.nexgo.utils.FormatUtils;
import cn.nexgo.utils.MixUtils;

/***************************************************************************************************
 *                                  Copyright (C), Nexgo Inc.                                      *
 *                                    http://www.nexgo.cn                                          *
 ***************************************************************************************************
 * usage           : 
 * Version         : 1
 * Author          : Truth
 * Date            : 2017/12/18
 * Modify          : create file
 **************************************************************************************************/
public class TransComb {
    private static boolean LOG_ENABLE = true;
    private Logger log = LoggerFactory.getLogger(TransComb.class.getSimpleName());

    public static final int ISO8583_OFFSET = 13;

    private byte[] sendBytes;

    private ICommu commu;
    private List<SocketObject.SocketConnectEntity> socketConnectEntitys;
    private MsgBean.BaseTransEntity requestBean;
    private MsgBean.BaseTransEntity responseBaseBean;
    private boolean disconnectAfterRecv;
    private int macKeyIndex;
    private boolean checkMacEn;
    //    private boolean incTransNo;
    private MsgModel.ModelListener listener;

    private AppPinPad pinPad;

    public TransComb() {
        pinPad = new AppPinPad();
    }

    public void doTrans(ICommu commu, List<SocketObject.SocketConnectEntity> socketConnectEntitys,
                        MsgBean.BaseTransEntity requestBean, MsgBean.BaseTransEntity responseBaseBean,
                        boolean disconnectAfterRecv, int macKeyIndex,
                        boolean checkMacEn, boolean incTransNo, MsgModel.ModelListener listener) {

        this.commu = commu;
        this.socketConnectEntitys = socketConnectEntitys;
        this.requestBean = requestBean;
        this.responseBaseBean = responseBaseBean;
        this.disconnectAfterRecv = disconnectAfterRecv;
        this.macKeyIndex = macKeyIndex;
        this.checkMacEn = checkMacEn;
        this.listener = listener;

        if (incTransNo) {   // only connect success can inc trans No.
            MsgModel.incTransNo();
        }

        sendBytes = this.requestBean.packageMsg();

        if (sendBytes == null) {
            listener.onConnectError(MsgModel.ParamsError);
            return;
        }

        if (checkMacEn) {
            pinPad.calcMac(macKeyIndex, Arrays.copyOfRange(sendBytes, ISO8583_OFFSET, sendBytes.length - 8), AppPinPad.MacAlgorithmEnum.ECB, macCalcListener);
            return;
        }

        startTrans();
    }

    private AppPinPad.CalcMacListenner macCalcListener = new AppPinPad.CalcMacListenner() {
        @Override
        public void onSuccess(byte[] macData) {
            System.arraycopy(macData, 0, sendBytes, sendBytes.length - 8, 8);
            startTrans();
        }

        @Override
        public void onFail(cn.nexgo.utils.ErrorCode errorCode) {
            listener.onConnectError(MsgModel.Nokey);
        }
    };

    // step 1: send & receive data
    private void startTrans() {
        log.debug("sendbytes>>>>-> " + ConvertUtils.bytes2HexString(sendBytes));
        traceIso8583Request(sendBytes);

//        sendBytes = ConvertUtils.hexString2Bytes("00 5A 60 00 02 00 00 60 31 00 31 34 56 08 00 00 20 00 00 00 C0 00 16 00 00 33 35 32 31 37 31 35 37 36 33 30 38 34 34 30 31 35 36 33 31 30 30 31 35 00 11 00 00 00 01 00 40 00 28 53 65 71 75 65 6E 63 65 20 4E 6F 31 35 30 30 30 30 4E 35 30 30 30 30 30 30 35 35 38 00 03 30 31 20");
        commu.exchangeMsg(socketConnectEntitys, sendBytes, disconnectAfterRecv, new Cn8583Unzip(), GData.getInstance().getSetupEntity().getCommuConnectTimeout() * 1000, new CommuObject.TransListener() {
            @Override
            public void onConnectError(CommuObject.ErrorCode errorCode) {
                log.error("onConnectError->" + errorCode.getMsg());
                listener.onConnectError(MsgModel.ConnectError);
            }

            @Override
            public void onConnectSuccess() {
                log.debug("onConnectSuccess");
                listener.onConnectSuccess();
            }

            @Override
            public void onSendError(CommuObject.ErrorCode errorCode) {
                log.error("onSendError->" + errorCode.getMsg());
                listener.onSendError(MsgModel.SendError);
            }

            @Override
            public void onSendSuccess() {
                log.debug("onSendSuccess");
                listener.onSendSuccess();
            }

            @Override
            public void onReceiveError(CommuObject.ErrorCode errorCode) {
                log.error("onReceiveError->" + errorCode.getMsg());
                listener.onReceiveError(new ErrorCode(MsgModel.RecvError.getCode(), errorCode.getMsg()));
            }

            @Override
            public void onReceiveSuccess(final byte[] data) {
                log.debug("onReceiveSuccess->" + ConvertUtils.bytes2HexString(data));

                analysisReceiveData(data);
            }

            @Override
            public void onCancel() {
                log.debug("onCancel");
            }
        });
    }

    // step 2: analysis receive data
    private void analysisReceiveData(byte[] data) {
        if (!responseBaseBean.parseMsg(data, 0)) {  // parse error
            log.error("8583 parse error");
            listener.onReceiveError(new ErrorCode(MsgModel.RecvError.getCode(), BaseUtils.getApp().getString(R.string.trans_error_8583_parse_error)));
            return;
        }
        setSystime(responseBaseBean);  // set system time

        if (!checkMacEn) {  // need not check mac
            neednotCheckMac();
            return;
        }

        needCheckMac(data);
    }

    // step 3.1: need check mac
    private void neednotCheckMac() {
        ErrorCode errorCode = checkParams(requestBean, responseBaseBean); // check response params
        if (errorCode != null) {
            log.error("response format error->" + errorCode.getMsg());
            listener.onReceiveError(errorCode);
            return;
        }

        listener.onReceiveSuccess(responseBaseBean);
    }

    // step 3.2: need check mac
    private void needCheckMac(byte[] orgData) {
        if ((responseBaseBean.getMac() == null) || (responseBaseBean.getMac().length() != 16)) {
            log.error("have no mac field");
            listener.onReceiveError(MsgModel.MACError);
            return;
        }


        pinPad.calcMac(macKeyIndex, Arrays.copyOf(orgData, orgData.length - 8),
                AppPinPad.MacAlgorithmEnum.ECB, receivMacListenner);
    }

    private AppPinPad.CalcMacListenner receivMacListenner = new AppPinPad.CalcMacListenner() {
        @Override
        public void onSuccess(byte[] macData) {
            if (!MixUtils.compare(ConvertUtils.hexString2Bytes(responseBaseBean.getMac()), macData)) {
                log.error("receivMac calc error");
                listener.onReceiveError(MsgModel.MACError);
                return;
            }
            ErrorCode errorCode = checkParams(requestBean, responseBaseBean); // check response params
            if (errorCode != null) {
                log.error("response format error->" + errorCode.getMsg());
                listener.onReceiveError(errorCode);
                return;
            }

            listener.onReceiveSuccess(responseBaseBean);
        }

        @Override
        public void onFail(cn.nexgo.utils.ErrorCode errorCode) {
            log.error("receivMac calc Fail->" + errorCode.getMsg());
            listener.onReceiveError(MsgModel.MACError);
        }
    };

    private ErrorCode checkParams(MsgBean.BaseTransEntity requestBean, MsgBean.BaseTransEntity responseBean) {
        if ((requestBean == null) || (responseBean == null)) {
            return new ErrorCode(MsgModel.ParamsError.getCode(), BaseUtils.getApp().getString(R.string.trans_responseparamserror));
        }

        // msg type
        String sourceStr = requestBean.getMsgType();
        String distStr = responseBean.getMsgType();
        if ((sourceStr == null) || (distStr == null)) {
            return new ErrorCode(MsgModel.ParamsError.getCode(), BaseUtils.getApp().getString(R.string.trans_responseparamserror) +
                    BaseUtils.getApp().getString(R.string.trans_responseparamserror_nomsgtype));
        }
        if ((Integer.parseInt(sourceStr) + 10) != Integer.parseInt(distStr)) {
            return new ErrorCode(MsgModel.ParamsError.getCode(), BaseUtils.getApp().getString(R.string.trans_responseparamserror_msgtypeerror));
        }

        // process code
        sourceStr = requestBean.getTransRecordEntity().getMsgProcCode();
        distStr = responseBean.getTransRecordEntity().getMsgProcCode();
        if ((sourceStr != null) && (distStr != null)) {
            if (Long.parseLong(sourceStr) != Long.parseLong(distStr)) {
                log.error("Long.parseLong(sourceStr) != Long.parseLong(distStr)->" + sourceStr + ", " + distStr);
                return new ErrorCode(MsgModel.ParamsError.getCode(), BaseUtils.getApp().getString(R.string.trans_responseparamserror_processcodeerror));
            }
        }

        // amount
        sourceStr = requestBean.getTransRecordEntity().getMsgAmount();
        distStr = responseBean.getTransRecordEntity().getMsgAmount();
        if ((sourceStr != null) && (distStr != null)) {
            if (Long.parseLong(sourceStr) != Long.parseLong(distStr)) {
                return new ErrorCode(MsgModel.ParamsError.getCode(), BaseUtils.getApp().getString(R.string.trans_responseparamserror_amounterror));
            }
        }

        // trans no
        sourceStr = requestBean.getTransRecordEntity().getMsgTraceNO();
        distStr = responseBean.getTransRecordEntity().getMsgTraceNO();
        if ((sourceStr != null) && (distStr != null)) {
            if (Long.parseLong(sourceStr) != Long.parseLong(distStr)) {
                return new ErrorCode(MsgModel.ParamsError.getCode(), BaseUtils.getApp().getString(R.string.trans_responseparamserror_traceno));
            }
        }

        // TID, MID
        sourceStr = requestBean.getTransRecordEntity().getMsgTermID();
        distStr = responseBean.getTransRecordEntity().getMsgTermID();
        if ((sourceStr != null) && (distStr != null)) {
            sourceStr = FormatUtils.appendArray(sourceStr, 8, true, ' ');
            if (!sourceStr.equals(distStr)) {
                log.error("TID error->" + sourceStr + ", " + distStr);
                return new ErrorCode(MsgModel.ParamsError.getCode(), BaseUtils.getApp().getString(R.string.trans_responseparamserror_tid));
            }
        }

        sourceStr = requestBean.getTransRecordEntity().getMsgMerchID();
        distStr = responseBean.getTransRecordEntity().getMsgMerchID();
        if ((sourceStr != null) && (distStr != null)) {
            sourceStr = FormatUtils.appendArray(sourceStr, 15, true, ' ');
            if (!sourceStr.equals(distStr)) {
                log.error("MID error->" + sourceStr + ", " + distStr);
                return new ErrorCode(MsgModel.ParamsError.getCode(), BaseUtils.getApp().getString(R.string.trans_responseparamserror_mid));
            }
        }

        // response code
//        distStr = responseBean.getTransRecordEntity().getMsgRespCode();
//        if (distStr == null) {
//            return new ErrorCode(MsgModel.ParamsError.getCode(), BaseUtils.getApp().getString(R.string.trans_responseparamserror) +
//                    BaseUtils.getApp().getString(R.string.trans_responseparamserror_noresponsecode));
//        }

        return null;
    }

    private void setSystime(MsgBean.BaseTransEntity responseBaseBean) {
        if ((responseBaseBean.getTransRecordEntity().getMsgTransTime() != null) &&
                (responseBaseBean.getTransRecordEntity().getMsgTransDate() != null)) {
            SysTime.setSystemTime(responseBaseBean.getTransRecordEntity().getMsgTransDate() + responseBaseBean.getTransRecordEntity().getMsgTransTime());
        }
    }

    private void traceIso8583Request(byte[] request) {
        if (!LOG_ENABLE) {
            return;
        }

        MessageFactory iso8583MsgFactory = null;
        try {
            InputStream is = BaseUtils.getApp().getAssets().open("iso8583.xml");
            iso8583MsgFactory = ConfigParser.creatFromStream(is);
            is.close();
        } catch (IOException e) {
            e.printStackTrace();
        }

        if (iso8583MsgFactory == null) {
            return;
        }

        Message msg = iso8583MsgFactory.parseMsg(request, ISO8583_OFFSET);
        if (msg == null) {
            log.error("msg format error");
            return;
        }

        log.debug("\nsend -->> {}", msg.toString());
    }

}
