package cn.nexgo.inbas.components.input.pin.internal;

import android.os.Handler;
import android.os.Looper;

import com.nexgo.oaf.apiv3.APIProxy;
import com.nexgo.oaf.apiv3.SdkResult;
import com.nexgo.oaf.apiv3.device.pinpad.OnPinPadInputListener;
import com.nexgo.oaf.apiv3.device.pinpad.PinAlgorithmModeEnum;
import com.nexgo.oaf.apiv3.device.pinpad.PinPad;
import com.nexgo.oaf.apiv3.device.pinpad.PinPadKeyCode;

import cn.nexgo.inbas.components.input.pin.InputPinData;
import cn.nexgo.utils.BaseUtils;

/***************************************************************************************************
 *                                  Copyright (C), Nexgo Inc.                                      *
 *                                    http://www.nexgo.cn                                          *
 ***************************************************************************************************
 * usage           : 
 * Version         : 1
 * Author          : zhouxie
 * Date            : 2017/12/12
 * Modify          : create file
 **************************************************************************************************/

public class InputPinPresenter implements InputPinContract.Presenter {

    private final Handler handler;
    private InputPinData inputPinData;
    private InputPinContract.View view;
    private final PinPad sdkPinPad;

    private boolean isRunning;

    public InputPinPresenter(InputPinContract.View view, InputPinData inputPinData) {
        this.view = view;
        this.inputPinData = inputPinData;
        sdkPinPad = APIProxy.getDeviceEngine(BaseUtils.getApp().getApplicationContext()).getPinPad();
        handler = new Handler(Looper.getMainLooper());
    }

    @Override
    public void startInputPin(InputPinData inputPinData) {
        if (isRunning) {
            return;
        }
        view.startTimeOutCountDown(inputPinData.getTimeout());

        int[] pinLimit = inputPinData.getPinlen();
        int ret;
        byte[] bytes = sdkPinPad.setPinpadLayout(view.getPinPadLayoutEntity());
        view.setPinNum(bytes);
        view.setPinLen(inputPinData.getPinlen());
        System.out.println("inputOnlinePin->"+ 1);
        if (!inputPinData.isOfflinePin()) {
            if((inputPinData.getPan() == null) ||
                    (inputPinData.getPan().length() < 13) ||
                    (inputPinData.getPan().length() > 19)){
                view.showPinError();
                return;
            }
            byte[] pan = inputPinData.getPan().getBytes();
            // FIXME: 2018/9/29 Hassan remark: N5 have no limit for timeout, but N3 the limit is 120s. so change the timeout to 99s
            ret = sdkPinPad.inputOnlinePin(pinLimit, 99, pan, inputPinData.getKeyIndex(), PinAlgorithmModeEnum.ISO9564FMT1, mPinPadInputListener);
        } else {
            ret = sdkPinPad.inputOfflinePin(pinLimit, 99, mPinPadInputListener);
        }
        System.out.println("inputOnlinePin->"+ret);
        if (ret == SdkResult.Success) {
            isRunning = true;
        } else {
            sdkPinPad.cancelInput();
            view.showPinError();
        }
    }

    @Override
    public void cancelInputPin() {
        if (sdkPinPad.isInputting() && isRunning) {
            sdkPinPad.cancelInput();
        }
    }

    private boolean isTimeOut = false;

    @Override
    public void onInputPinTimeOut() {
        isTimeOut = true;
        if (sdkPinPad.isInputting() && isRunning) {
            sdkPinPad.cancelInput();
        }
    }

    private OnPinPadInputListener mPinPadInputListener = new OnPinPadInputListener() {
        @Override
        public void onInputResult(int retCode, final byte[] data) {
            isRunning = false;
            System.out.println("onInputResult->"+retCode);
            switch (retCode) {
                case SdkResult.Success:
                    postMainThread(new Runnable() {
                        @Override
                        public void run() {
                            view.showPinConfirm(data);
                        }
                    });
                    break;
                case SdkResult.PinPad_No_Pin_Input:
                    postMainThread(new Runnable() {
                        @Override
                        public void run() {
//                            view.showPinConfirm(null);
                            view.pinPass();
                        }
                    });
                    break;
                case SdkResult.TimeOut:
                    postMainThread(new Runnable() {
                        @Override
                        public void run() {
                            view.showPinTimeOut();
                        }
                    });
                    break;
                case SdkResult.PinPad_Input_Cancel:
                    postMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (isTimeOut) {
                                view.showPinTimeOut();
                                isTimeOut = false;
                            } else {
                                view.showPinCancel();
                            }
                        }
                    });
                    break;
                default:
                    postMainThread(new Runnable() {
                        @Override
                        public void run() {
                            view.showPinError();
                        }
                    });
                    break;
            }
        }

        @Override
        public void onSendKey(byte keyCode) {
            System.out.println("onSendKey->"+keyCode);
            if (keyCode == PinPadKeyCode.KEYCODE_CLEAR) {
                postMainThread(new Runnable() {
                    @Override
                    public void run() {
                        view.showPinClear();
                    }
                });
            } else {
                postMainThread(new Runnable() {
                    @Override
                    public void run() {
                        view.showPinMask();
                    }
                });
            }
        }
    };

    private void postMainThread(Runnable runnable) {
        if (view.isActive()) {
            if (Looper.myLooper() == Looper.getMainLooper()) {
                runnable.run();
            } else {
                handler.post(runnable);
            }
        }
    }
}
