package cn.nexgo.inbas.components.data.repository;

import org.greenrobot.greendao.DaoException;
import org.greenrobot.greendao.query.QueryBuilder;

import java.util.List;

import cn.nexgo.inbas.components.data.DataConstant;
import cn.nexgo.inbas.components.data.DataOpenHelper;
import cn.nexgo.inbas.components.data.daoGenerate.DaoSession;
import cn.nexgo.inbas.components.data.daoGenerate.OperatorEntityDao;
import cn.nexgo.inbas.components.data.bean.OperatorEntity;
import cn.nexgo.inbas.components.data.helper.SettingDbHelper;
import cn.nexgo.utils.AppLogUtils;

/***************************************************************************************************
 *                                  Copyright (C), Nexgo Inc.                                      *
 *                                    http://www.nexgo.cn                                          *
 ***************************************************************************************************
 * File Name     : OperatorRepository.java
 * Usage         : Repository of operator DAO
 * Version       : 1
 * Author        : Brad
 * Date          : 2017/12/13
 * Modification  : Created file

 **************************************************************************************************/
public class OperatorRepository{
    private static String LOG_TAG = "OperatorRepository";
    private OperatorEntityDao recordEntityDao;
    private static volatile OperatorRepository INSTANCE;
    private OperatorRepository() {
        init();
    }
    public static OperatorRepository getInstance(){
        if (INSTANCE == null){
            synchronized (OperatorRepository.class){
                if (INSTANCE == null){
                    INSTANCE = new OperatorRepository();
                }
            }
        }
        return INSTANCE;
    }

    /**
     *  Initialize Entity Dao
     */
    private void init(){
        DaoSession daoSession = SettingDbHelper.getInstance().getDaoSession();
        recordEntityDao = daoSession.getOperatorEntityDao();
    }


    /**
     * Insert an Entity. If id is already existed, then update the existed record.
     * @param entity Entity
     * @return Success & fail
     */
    public boolean insert(OperatorEntity entity) {
        if(entity == null){
            return false;
        }
        if (recordEntityDao == null){
            init();
        }
        if (recordEntityDao.count() >= DataConstant.MAX_CASHIER_NUM){
            AppLogUtils.warn(true, LOG_TAG,  "Operator records are bigger than {}", DataConstant.MAX_CASHIER_NUM);
            return false;
        }

        recordEntityDao.insert(entity);
        return true;
    }

    /**
     * Delete an Entity
     * @param entity Entity
     * @return Success & fail
     */
    public boolean delete(OperatorEntity entity) {
        if(entity == null){
            return false;
        }
        if (recordEntityDao == null){
            init();
        }

        try{
            recordEntityDao.delete(entity);
        }catch (DaoException e){
            AppLogUtils.error(true, LOG_TAG,  "Delete failed throw Dao exception");
            return false;
        }

        return true;
    }

    /**
     * Delete Last Entity
     * @return Success & fail
     */
    public boolean deleteLast() {
        if (recordEntityDao == null){
            init();
        }
        recordEntityDao.detachAll();
        QueryBuilder<OperatorEntity> qb = recordEntityDao.queryBuilder().orderDesc(OperatorEntityDao.Properties.Id).limit(1);    //invert list
        List<OperatorEntity> gotRecords = qb.build().list();      //Get Object List
        if (gotRecords == null || gotRecords.isEmpty()) {
            return true;
        }
        delete(gotRecords.get(0));
        return true;
    }

    /**
     * Delete All Entities
     */
    public void deleteAll() {
        if (recordEntityDao == null){
            init();
        }
        recordEntityDao.deleteAll();
    }

    /**
     * Get Last Entity
     * @return Entity
     */
    public OperatorEntity getLast() {
        if (recordEntityDao == null) {
            init();
        }
        recordEntityDao.detachAll();
        QueryBuilder<OperatorEntity> qb = recordEntityDao.queryBuilder().orderDesc(OperatorEntityDao.Properties.Id).limit(1);    //invert list
        return qb.build().unique();
    }

    /**
     * Update an Entity. If id is not existed, insert a new entity.
     * @param entity Entity
     * @return Success & fail
     */
    public boolean update(OperatorEntity entity) {
        if(entity == null){
            return false;
        }

        if (recordEntityDao == null){
            init();
        }

        recordEntityDao.update(entity);
        return true;
    }

    /**
     * Get Record Nunbers
     * @return Record Numbers
     */
    public int getRecordNum() {
        if (recordEntityDao == null){
            init();
        }
        AppLogUtils.trace(DataOpenHelper.LOG_ENABLE, LOG_TAG, "getRecordNum = "+ (int) recordEntityDao.count());
        return (int)recordEntityDao.count();
    }

    /**
     * Get all operator records
     * @return List<OperatorEntity>
     */
    public List<OperatorEntity> getAll(){
        if (recordEntityDao == null){
            init();
        }
        recordEntityDao.detachAll();
        QueryBuilder<OperatorEntity> qb = recordEntityDao.queryBuilder().limit(DataConstant.MAX_CASHIER_NUM);    //invert list
        return qb.build().list();              //Get List
    }

    /**
     * Get an Entity by operator ID
     * @param operatorNo
     * @return Entity with the specified ID
     */
    public OperatorEntity getByOperatorNo(String operatorNo) {
        if (recordEntityDao == null){
            init();
        }
        recordEntityDao.detachAll();
        QueryBuilder<OperatorEntity> qb = recordEntityDao.queryBuilder();
        qb.where(OperatorEntityDao.Properties.OperatorNo.eq(operatorNo))
                .limit(1);     // Limit
        return qb.build().unique();
    }

}
